# vim: set foldmethod=marker foldmarker={{{,}}} :
var displayClass = {
    new: func(device) {
# the contructor {{{
        var m = { parents: [ displayClass ] };

        m.display = canvas.new({
                "name"      : device.name,
                "size"      : [1280, 1280],
                "view"      : [1024, 768],
                "mipmapping": 1
        });
        m.display.addPlacement({
                "node": device.data['screen-object'] != nil ? device.data['screen-object'] : "Screen",
                "parent": device.name
            });
        m.display.setColorBackground(0,0,0);
        m.role = device.role;
        m.device = device;
        m.screenElements = {};
        m.screen = m.display
            .createGroup()
            .show();

        m.timers2 = {}; # tho old timer implementation use already a named timer hash
        # Softkeys revert to the previous level after 45 seconds of inactivity.
        m.softkeys_inactivity_delay = 45;

            var groups = {
                show : [
                    'Header',
                    'SoftKeysTexts', 
                    'COMM', 
                    'NAV', 
                    'nav-freq-switch',
                    'comm-freq-switch',
                ],
                text: [
                    'nav1-standby-freq', 'nav1-selected-freq', 'nav1-id',
                    'nav2-standby-freq', 'nav2-selected-freq', 'nav2-id',
                    'comm1-standby-freq', 'comm1-selected-freq',
                    'comm2-standby-freq', 'comm2-selected-freq',
                ],
                hide : [ 'Failures', 'NAV-COMM-Failures' ],
                clip: [ ],
            };

            for (var k = 0; k < 12; k += 1) {
                append(groups.text, sprintf("SoftKey%02i-text", k));
                append(groups.show, sprintf("SoftKey%02i-bg", k));
            }

            if (m.device.role == 'PFD') {
                append(groups.show,
# {{{
                    'XPDR-TIME', 
                    'FlightInstruments',
                    'Horizon',
                    'bankPointer',
                    'VSI',
                    'Rose',
                    'Heading-bug',
                    'PFD-Widgets',
                    'Trends',
                    'Airspeed-Trend-Indicator',
                    'Altitude-Trend-Indicator',
                    'OAT',
                    'IAS-bg',
                    'TAS',
                    'GSPD',
                    'BARO-bg',
                    'SlipSkid',
                    'IAS-Vx', 'IAS-Vy', 'IAS-Vr', 'IAS-Vglide',
# }}}
                );
                append(groups.hide,
# {{{
                    'EIS',
                    'CDI',
                    'OMI', 'MarkerBG', 'MarkerText',
                    'VDI',
                    'NAV1-pointer', 'NAV1-CDI', 'NAV1-FROM', 'NAV1-TO',
                    'NAV2-pointer', 'NAV2-CDI', 'NAV2-FROM', 'NAV2-TO',
                    'GPS-pointer', 'GPS-CDI', 'GPS-CTI', 'GPS-CTI-diamond', 'GPS-FROM', 'GPS-TO',
                    'BRG1-pointer',
                    'BRG2-pointer',
                    'SelectedHDG-bg',
                    'SelectedHDG-bgtext',
                    'SelectedHDG-text',
                    'SelectedCRS-bg',
                    'SelectedCRS-bgtext',
                    'SelectedCRS-text',
                    'SelectedALT', 'SelectedALT-bug', 'SelectedALT-bg', 'SelectedALT-symbol',
                    'TAS',
                    'GSPD',
                    'WindData',
                    'Reversionnary',
                    'Annunciation',
                    'Comparator',
                    'BRG1',
                    'BRG2',
                    'DME1',
                    'PFD-Map-bg',
                    'PFD-Multilines',
                    'MapOrientation',
                    'WindData', 'WindData-OPTN1', 'WindData-OPTN2', 'WindData-OPTN1-HDG', 'WindData-OPTN2-symbol', 'WindData-OPTN2-headwind', 'WindData-OPTN2-crosswind', 'WindData-NODATA',
                    'AOA', 'AOA-needle', 'AOA-text', 'AOA-approach',
                    'MFD-navbox',
                    'Traffic',
# }}}
                );
                append(groups.clip,
# {{{
                    'SpeedLint1',
                    'SpeedTape',
                    'LintAlt',
                    'AltLint00011',
                    'PitchScale',
# }}}
                );
                append(groups.text,
# {{{
                    'SelectedALT-text',
                    'TAS-text',
                    'GSPD-text',
                    'TIME-text',
                    'OAT-text',
                    'VSIText',
                    'Speed110',
                    'Alt11100',
                    'HDG-text',
                    'BARO-text',
                    'CDI-SRC-text', 'CDI-GPS-ANN-text', 'CDI-GPS-XTK-text',
                    'BRG1-pointer', 'BRG1-SRC-text', 'BRG1-DST-text', 'BRG1-WPID-text',
                    'BRG2-pointer', 'BRG2-SRC-text', 'BRG2-DST-text', 'BRG2-WPID-text',
                    'WindData-OPTN1-HDG-text', 'WindData-OPTN1-SPD-text',
                    'WindData-OPTN2-crosswind-text', 'WindData-OPTN2-headwind-text',
                    'XPDR-MODE-text', 'XPDR-DIGIT-3-text', 'XPDR-DIGIT-2-text', 'XPDR-DIGIT-1-text', 'XPDR-DIGIT-0-text',
                    'ETE', 'ETE-text', 'DIS', 'DIS-text', 'LEG-text',
                    'AltBigC', 'AltSmallC'
# }}}
                );
                for (var place = 1; place <= 6; place +=1) {
                    append(groups.text,
                        'AltBigU' ~ place,
                        'AltSmallU' ~ place,
                        'AltBigD' ~ place,
                        'AltSmallD' ~ place
                    );
                }
                canvas.parsesvg(m.screen, data.zkv1000_reldir ~ 'Models/PFD.svg');
            }
            else {
                var eis_file = getprop('/instrumentation/zkv1000/eis/type');
                if (eis_file == nil)
                    eis_file = getprop('/instrumentation/zkv1000/eis/file');

                if (eis_file != nil) {
                    if (find('/', eis_file) == -1)
                        eis_file = data.zkv1000_dir ~ 'Nasal/EIS/' ~ eis_file ~ '.nas';
                    elsif (split('/', eis_file)[0] == 'Aircraft') {
                        var path = split('/', eis_file);
                        if (getprop('/sim/fg-aircraft') != nil) {
                            eis_file = getprop('/sim/fg-aircraft');
                            for (var i = 1; i < size(path); i += 1)
                                eis_file ~= '/' ~ path[i];
                        }
                        else
                            eis_file = getprop('/sim/fg-root') ~ '/' ~ eis_file;
                    }
                }
                else
                    eis_file = data.zkv1000_dir ~ 'Nasal/EIS/none.nas';

                if (io.stat(eis_file) == nil
                and print(eis_file ~ ' not found'))
                    eis_file = data.zkv1000_dir ~ 'Nasal/EIS/none.nas';
                io.load_nasal(eis_file, 'zkv1000');

                if (contains(m.parents[0], 'showEIS'))
                    m.showEIS(groups);

                append(groups.hide, 'PFD-navbox');
                for (var i=1; i <= 4; i+=1)
                    foreach (var t; ['ID', 'VAL'])
                        append(groups.text, 'DATA-FIELD' ~ i ~ '-' ~ t ~ '-text');
            }

            canvas.parsesvg(m.screen, data.zkv1000_reldir ~ 'Models/softkeys.svg');
            canvas.parsesvg(m.screen, data.zkv1000_reldir ~ 'Models/header-nav-comm.svg');

            m.loadGroup(groups);

            if (m.device.role == 'PFD') {
                m.device.data.aoa = 0;
                m.device.data['aoa-auto'] = 0;
                m.device.data.mapclip = {
                    top: math.ceil(m.screenElements['PFD-Map-bg'].getTransformedBounds()[1]) - 1,
                    right: math.ceil(m.screenElements['PFD-Map-bg'].getTransformedBounds()[2]) - 1,
                    bottom: math.ceil(m.screenElements['PFD-Map-bg'].getTransformedBounds()[3]) - 1,
                    left: math.ceil(m.screenElements['PFD-Map-bg'].getTransformedBounds()[0]) - 1,
                };
            }
            else {
                m.device.data.mapclip = {
                    top: math.ceil(m.screenElements['Header'].getTransformedBounds()[3]),
                    right: m.display.get('view[0]'),
                    bottom: math.ceil(m.screenElements['SoftKeysTexts'].getTransformedBounds()[1]),
                    left: contains(m.screenElements, 'EIS') ? math.ceil(m.screenElements['EIS'].getTransformedBounds()[2]) : 0,
                };
            }
        return m;
    },
#}}}

# timers stuff {{{
    timers : {},

    timerTrigger : func {
        var now = systime();
        foreach (var id; keys(me.timers)) {
            if (me.timers[id] < now) {
                me.screenElements[id].hide();
                delete(me.timers, id);
            }
        }
        settimer(func me.timerTrigger(), 1);
    },

    addTimer : func (duration, element) {
        if (typeof(element) == 'scalar')
            element = [ element ];
        var end = systime() + duration;
        foreach (var e; element)
            me.timers[e] = end;
    },
#}}}

    showInitProgress : func {
#{{{
        if (me.device.role == 'PFD') {
            me.updateAI();
            me.updateVSI();
            me.updateIAS();
            me.updateALT();
            me.updateHSI();
            me.updateTIME();
            me.updateOAT();
            me.updateTAS();
            me.updateBRG();
            me.updateXPDR();
            me.updateBARO();
            me.updateOMI();
            me.timerTrigger();
            me.screen.show();
        }
        else {
            me.updateEIS();
            io.load_nasal(data.zkv1000_dir ~ 'Nasal/MFD.pages.nas', 'zkv1000');
            me['page selected'] = 0;
            me.setMFDPages();
            me.device.buttons.MENU = me.device.buttons.MapMenu;
        }

        me.updateNAV({auto:'nav', tune: radios.getNode('nav-tune').getValue()});
        me.updateCOMM({auto:'comm', tune: radios.getNode('comm-tune').getValue()});
        me.softkeys_inactivity();
        me.updateSoftKeys();
        me.updateNavigationBox();
    },
#}}}

    colors : {
# set of predefined colors {{{
        green : [0, 1, 0],
        white : [1, 1, 1],
        black : [0, 0, 0],
        lightblue : [0, 1, 1],
        darkblue : [0, 0, 1],
        red : [1, 0, 0],
        magenta : [1, 0, 1],
    },
#}}}

    loadGroup : func (h) {
#{{{
        if (typeof(h) != 'hash') {
            msg_dbg(sprintf("%s need a hash, but get a %s from %s",
                    caller(0)[0],
                    typeof(h),
                    caller(1)[0]));
            return;
        }
        var setMethod = func (e, t) {
            if (t == 'hide')
                me.screenElements[e].hide();
            elsif (t == 'show')
                me.screenElements[e].show();
            elsif (t == 'rot' or t == 'trans') {
                if (! contains(me.screenElements[e], t))
                    me.screenElements[e][t] = me.screenElements[e].createTransform();
            }
            elsif (t == 'clip') {
                if (contains(me.clips, e))
                    me.screenElements[e].set("clip", me.clips[e]);
                else
                    printlog('warn', 'no defined clip for ' ~ e);
            }
            elsif (t == 'text') {
                if (contains(me.texts, e)) {
                    if (contains(me.texts[e], 'alignment'))
                        me.screenElements[e].setAlignment(me.texts[e].alignment);
                    if (contains(me.texts[e], 'default'))
                        me.screenElements[e].setText(me.texts[e].default);
                    if (contains(me.texts[e], 'color'))
                        me.screenElements[e].setColor(me.texts[e].color);
                    if (contains(me.texts[e], 'visible'))
                        me.screenElements[e].setVisible(me.texts[e].visible);
                }
                else
                    printlog('debug', 'no text format for ' ~ e);
            }
            else
                printlog('warn', 'unknown method ' ~ t);
        };
        foreach (var todo; keys(h)) {
            if (typeof(h[todo]) != 'vector') h[todo] = [ h[todo] ];
            foreach (var id; h[todo]) {
                if (! contains(me.screenElements, id)) {
                    me.screenElements[id] = me.screen.getElementById(id);
                    if (me.screenElements[id] != nil)
                        setMethod(id, todo);
                    else
                        printlog('warn', 'SVG ID ' ~ id ~ ' not found');
                }
                else
                    setMethod(id, todo);
            }
        }
    },
#}}}

    clips : {
#{{{
        PitchScale   : "rect(70,664,370,256)",
        SpeedLint1   : "rect(252,226,318,204)",
        SpeedTape    : "rect(115,239,455,156)",
        LintAlt      : "rect(115,808,455,704)",
        AltLint00011 : "rect(252,804,318,771)",
    },
#}}}

    texts : {
#{{{
        VSIText : {
            alignment: "right-bottom", default : num('0'),
        },
        Speed110 : {
            alignment : 'left-bottom'
        },
        Alt11100 : {
            alignment:'left-bottom'
        },
        "HDG-text" : {
            default: '---°'
        },
        'nav1-standby-freq' : {
            color: [0, 1, 1],
        },
        'nav1-id' : {
            default: ''
        },
        'nav2-id' : {
            default: ''
        },
        'CDI-GPS-ANN-text' : {
            visible : 0
        },
        'CDI-GPS-XTK-text' : {
            visible : 0
        },
        'CDI-SRC-text' : {
            visible : 0
        },
        'BARO-text' : {
            alignment : 'left-bottom',
        }
#        'TAS-text' : {
#            alignment : 'right-bottom',
#        },
#        'GSPD-text' : {
#            alignment : 'right-bottom',
#        },
    },
#}}}

    softkeys_inactivity : func {
# automagically back to previous level after some delay {{{
        me.timers2.softkeys_inactivity = maketimer (
            me.softkeys_inactivity_delay,
            func {
                pop(me.device.softkeys.path);
                me.updateSoftKeys();
            }, me);
        me.timers2.softkeys_inactivity.singleShot = 1;
    },
#}}}

    setSoftKeyColor : func (n, active, implemented = 1, alert = 0) {
# set colors for active softkeys {{{
        var sftk = sprintf('SoftKey%02i-', n);
        if (active) {
            var bg = alert ? 1 : 0.5;
            me.screenElements[sftk ~ 'bg']
                .setColorFill(bg,bg,bg);
            me.screenElements[sftk ~ 'text']
                .setColor(0,0,0);
        }
        else {
            var tc = implemented ? 1 : 0.5;
            me.screenElements[sftk ~ 'bg']
                .setColorFill(0,0,0);
            me.screenElements[sftk ~ 'text']
                .setColor(tc,tc,tc);
        }
    },
#}}}

    updateNavigationBox : func (getDataFields = 0) {
# update Navigation Box on MFD and PFD header {{{
        var route = me.device.map.layers.route;
        var fpSize = size(route.flightPlan);
        var navbox = {
            DTK: func {
                var dtk = getprop('/instrumentation/gps/wp/wp[1]/desired-course-deg');
                if (dtk == nil)
                    return '---°';
                else
                    return sprintf('%03i°', dtk);
            },
            ETE: func {
                if (fpSize == 0) return '--:--';
                var eteSeconds = getprop('/autopilot/route-manager/ete');
                var eteHours = math.floor(eteSeconds / 3600);
                var eteMinutes = int((eteSeconds - (eteHours * 3600)) / 60);
                return sprintf(eteHours > 99 ? '--:--' : '%02i:%02i', eteHours, eteMinutes);
            },
            DIS: func {
                if (fpSize == 0) return '---NM';
                var dist = getprop('/autopilot/route-manager/distance-remaining-nm');
                return sprintf(dist >= 100 ? '%i' : '%.1f', dist);
            },
            LEG: func {
                if (fpSize == 0) return '';
                var wp = route.flightPlan[route.currentLeg.index];
                return wp[0].name ~ ' > ' ~ wp[1].name;
            },
            GS : func return sprintf('%iKT', getprop('/velocities/groundspeed-kt')),
            TRK: func return sprintf('%03i°', getprop('/orientation/track-deg')),
            TAS: func return sprintf('%i', getprop('/instrumentation/airspeed-indicator/true-speed-kt')),
            BRG: func return '---°',
            END: func return '---NM',
            ESA: func return '-----',
            ETA: func return '--:--',
            FOB: func return '---lbs',
            ISA: func return '-----',
            LDG: func return '--:--',
            MSA: func return '----ft',
            TKE: func return '---°',
            VSR: func return '----',
            XTK: func return '---NM',
        };
        if (getDataFields)
            return keys(navbox);
        if (me.device.role == 'MFD') {
            for (var i=1; i<=4; i+=1)
                me.screenElements['DATA-FIELD' ~ i ~ '-VAL-text']
                    .setText(
                        navbox[me.screenElements['DATA-FIELD' ~ i ~ '-ID-text'].get('text')]()
                    );
            settimer(func me.updateNavigationBox(), 0.6);
        }
        else { # PFD
            me.screenElements['ETE-text'].setText(navbox.ETE());
            me.screenElements['DIS-text'].setText(navbox.DIS());
            me.screenElements['LEG-text'].setText(navbox.LEG());
            settimer(func me.updateNavigationBox(), 0.3);
        }
    },
#}}}

    updateSoftKeys : func {
# update SoftKeys boxes {{{
        # on PFD the last boxes are always BACK and ALERTS
        if (me.device.role == 'PFD') {
            me.screenElements[sprintf("SoftKey%02i-text", 11)]
                .setText('ALERTS');
            if (size(me.device.softkeys.path) != 0)
                me.screenElements[sprintf("SoftKey%02i-text", 10)]
                    .setText('BACK')
                    .setColor(1,1,1);
        }

        var path = keyMap[me.device.role];
        var bindings = me.device.softkeys.bindings[me.device.role];
        var pathid = '';
        foreach (var p; me.device.softkeys.path) {
            path = path[p];
            pathid ~= p;
            if (contains(bindings, p))
                bindings = bindings[p];
        }

        # feeding with empty menus the first boxes
        var start = (contains(path, 'first')) ? path.first : 0;
        for (var k = 0; k < start; k+=1) {
            var sftk = sprintf("SoftKey%02i-", k);
            me.screenElements[sftk ~ 'text']
                .setText('');
            me.screenElements[sftk ~ 'bg']
                .setColorFill(0,0,0);
        }
        # filling with the content the next boxes
        forindex (var k; path.texts) {
            var i = k + start;
            me.screenElements[sprintf("SoftKey%02i-text", i)]
                .setText(path.texts[k]);
            me.setSoftKeyColor(i,
                    contains(me.device.softkeys.colored, pathid ~ path.texts[k]),
                    contains(bindings, path.texts[k]));
        }
        # feeding the last boxes with empty string
        var end = (me.device.role == 'PFD') ? 10 : 12;
        if (size(path.texts) + start < end) {
            start = size(path.texts) + start;
            for (var k = start; k < end; k += 1) {
                var sftk = sprintf("SoftKey%02i-", k);
                me.screenElements[sftk ~ 'text']
                    .setText('');
                me.screenElements[sftk ~ 'bg']
                    .setColorFill(0,0,0);
            }
        }

        if (size(me.device.softkeys.path))
            me.timers2.softkeys_inactivity.restart(me.softkeys_inactivity_delay);
        else
            me.timers2.softkeys_inactivity.stop();
    },
#}}}

    updateAI: func(){
#{{{
        var pitch = data.pitch;
        var roll = data.roll;
        if (pitch > 80)
            pitch = 80;
        elsif (pitch < -80)
            pitch = -80;
        me.screenElements.Horizon
            .setCenter(459, 282.8 - 6.849 * pitch)
            .setRotation(-roll * D2R)
            .setTranslation(0, pitch * 6.849);
        me.screenElements.bankPointer
            .setRotation(-roll * D2R);
        me.screenElements['SlipSkid']
            .setTranslation(getprop("/instrumentation/slip-skid-ball/indicated-slip-skid") * 10, 0);
        me.screenElements['Traffic']
            .setVisible(size(data.tcas));
        settimer(func me.updateAI(), 0.1);
    },
#}}}

    updateVSI: func () {
# animate VSI {{{
        var vsi = data.vsi;
        me.screenElements.VSIText
            .setText(num(math.round(vsi, 10)));
        if (vsi > 4500)
            vsi = 4500;
        elsif (vsi < -4500)
            vsi = -4500;
        me.screenElements.VSI
            .setTranslation(0, vsi * -0.03465);
        settimer(func me.updateVSI(), 0.1);
    },
#}}}

    updateIAS: func () {
# animates the IAS lint (PFD) {{{
        var ias = data.ias;
        if (ias >= 10)
            me.screenElements.Speed110
                .setText(sprintf("% 2u",num(math.floor(ias/10))));
        else
            me.screenElements.Speed110
                .setText('');
        me.screenElements.SpeedLint1
            .setTranslation(0,(math.mod(ias,10) + (ias >= 10)*10) * 36);
        me.screenElements.SpeedTape
            .setTranslation(0,ias * 5.711);
        if (ias > data.Vne and ! me._ias_already_exceeded) { # easier than .getColorFill
            me._ias_already_exceeded = 1;
            me.screenElements['IAS-bg']
                .setColorFill(1,0,0);
        }
        elsif (ias < data.Vne and me._ias_already_exceeded) { # easier than .getColorFill
            me._ias_already_exceeded = 0;
            me.screenElements['IAS-bg']
                .setColorFill(0,0,0);
        }
        foreach (var v; ['Vx', 'Vy', 'Vr', 'Vglide']) {
            if (me.device.data[v ~ '-visible'] and abs(data[v] - ias) < 30)
                me.screenElements['IAS-' ~ v]
                    .setTranslation(0, (ias - data[v]) * 5.711)
                    .show();
            else
                me.screenElements['IAS-' ~ v]
                    .hide();
        }

        var now = systime();
        # estimated speed in 6s
        var Sy = 6 * (ias - me._last_ias_kt) / (now - me._last_ias_s);
        if (abs(Sy) > 30)
            Sy = 30 * abs(Sy)/Sy; # = -30 or 30
        me.screenElements['Airspeed-Trend-Indicator']
            .setScale(1,Sy)
            .setTranslation(0, -284.5 * (Sy - 1));
        me._last_ias_kt = ias;
        me._last_ias_s = now;
        settimer(func me.updateIAS(), 0.1);
    },
    _last_ias_kt : 0,
    _last_ias_s : systime(),
    _ias_already_exceeded : 0,
#}}}

    updateTAS: func {
# updates the True Airspeed and GroundSpeed indicators {{{
        me.screenElements['TAS-text']
            .setText(sprintf('%iKT', getprop('/instrumentation/airspeed-indicator/true-speed-kt')));
        me.screenElements['GSPD-text']
            .setText(sprintf('%iKT', getprop('/velocities/groundspeed-kt')));
        settimer(func me.updateTAS(), 0.5);
    },
#}}}

    updateALT: func () {
# animates the altitude lint (PFD) {{{
        var alt = data.alt;
        if (alt < 0)
            me.screenElements.Alt11100
                .setText(sprintf("% 3i",math.ceil(alt/100)));
        elsif (alt < 100)
            me.screenElements.Alt11100
                .setText('');
        else
            me.screenElements.Alt11100
                .setText(sprintf("% 3i",math.floor(alt/100)));
        me.screenElements.AltLint00011
            .setTranslation(0,math.fmod(alt,100) * 1.24);

        # From Farmin/G1000 http://wiki.flightgear.org/Project_Farmin/FG1000
        if (alt> -1000 and alt< 1000000) {
            var Offset10 = 0;
            var Offset100 = 0;
            var Offset1000 = 0;
            if (alt< 0) {
                var Ne = 1;
                var alt= -alt;
            }
            else
                var Ne = 0;

            var Alt10       = math.mod(alt,100);
            var Alt100      = int(math.mod(alt/100,10));
            var Alt1000     = int(math.mod(alt/1000,10));
            var Alt10000    = int(math.mod(alt/10000,10));
            var Alt20       = math.mod(Alt10,20)/20;
            if (Alt10 >= 80)
                var Alt100 += Alt20;

            if (Alt10 >= 80 and Alt100 >= 9)
                var Alt1000 += Alt20;

            if (Alt10 >= 80 and Alt100 >= 9 and Alt1000 >= 9)
                var Alt10000 += Alt20;

            if (alt> 100)
                var Offset10 = 100;

            if (alt> 1000)
                var Offset100 = 10;

            if (alt> 10000)
                var Offset1000 = 10;

            if (!Ne) {
                me.screenElements.LintAlt.setTranslation(0,(math.mod(alt,100))*0.57375);
                var altCentral = (int(alt/100)*100);
            }
            elsif (Ne) {
                me.screenElements.LintAlt.setTranslation(0,(math.mod(alt,100))*-0.57375);
                var altCentral = -(int(alt/100)*100);
            }
            me.screenElements["AltBigC"].setText("");
            me.screenElements["AltSmallC"].setText("");
            for (var place = 1; place <= 6; place += 1) {
                var altUP = altCentral + (place*100);
                var offset = -30.078;
                if (altUP < 0) {
                    var altUP = -altUP;
                    var prefix = "-";
                    var offset += 15.039;
                }
                else
                    var prefix = "";

                if (altUP == 0) {
                    var AltBigUP    = "";
                    var AltSmallUP  = "0";

                }
                elsif (math.mod(altUP,500) == 0 and altUP != 0) {
                    var AltBigUP    = sprintf(prefix~"%1d", altUP);
                    var AltSmallUP  = "";
                }
                elsif (altUP < 1000 and (math.mod(altUP,500))) {
                    var AltBigUP    = "";
                    var AltSmallUP  = sprintf(prefix~"%1d", int(math.mod(altUP,1000)));
                    var offset = -30.078;
                }
                elsif ((altUP < 10000) and (altUP >= 1000) and (math.mod(altUP,500))) {
                    var AltBigUP    = sprintf(prefix~"%1d", int(altUP/1000));
                    var AltSmallUP  = sprintf("%1d", int(math.mod(altUP,1000)));
                    var offset += 15.039;
                }
                else {
                    var AltBigUP    = sprintf(prefix~"%1d", int(altUP/1000));
                    var mod = int(math.mod(altUP,1000));
                    var AltSmallUP  = sprintf("%1d", mod);
                    var offset += 30.078;
                }

                me.screenElements["AltBigU"~place].setText(AltBigUP);
                me.screenElements["AltSmallU"~place].setText(AltSmallUP);
                me.screenElements["AltSmallU"~place].setTranslation(offset,0);
                var altDOWN = altCentral - (place*100);
                var offset = -30.078;
                if (altDOWN < 0) {
                    var altDOWN = -altDOWN;
                    var prefix = "-";
                    var offset += 15.039;
                }
                else
                    var prefix = "";

                if (altDOWN == 0) {
                    var AltBigDOWN  = "";
                    var AltSmallDOWN    = "0";
                }
                elsif (math.mod(altDOWN,500) == 0 and altDOWN != 0) {
                    var AltBigDOWN  = sprintf(prefix~"%1d", altDOWN);
                    var AltSmallDOWN    = "";
                }
                elsif (altDOWN < 1000 and (math.mod(altDOWN,500))) {
                    var AltBigDOWN  = "";
                    var AltSmallDOWN    = sprintf(prefix~"%1d", int(math.mod(altDOWN,1000)));
                    var offset = -30.078;
                }
                elsif ((altDOWN < 10000) and (altDOWN >= 1000) and (math.mod(altDOWN,500))) {
                    var AltBigDOWN  = sprintf(prefix~"%1d", int(altDOWN/1000));
                    var AltSmallDOWN    = sprintf("%1d", int(math.mod(altDOWN,1000)));
                    var offset += 15.039;
                }
                else {
                    var AltBigDOWN  = sprintf(prefix~"%1d", int(altDOWN/1000));
                    var mod = int(math.mod(altDOWN,1000));
                    var AltSmallDOWN    = sprintf("%1d", mod);
                    var offset += 30.078;
                }
                me.screenElements["AltBigD"~place].setText(AltBigDOWN);
                me.screenElements["AltSmallD"~place].setText(AltSmallDOWN);
                me.screenElements["AltSmallD"~place].setTranslation(offset,0);
            }
        }
        me.updateSelectedALT();
        var now = systime();
        # altitude in 6s
        var Sy = .3 * (alt - me._last_alt_ft) / (now - me._last_alt_s); # scale = 1/20ft
        if (abs(Sy) > 15)
            Sy = 15 * abs(Sy)/Sy; # = -15 or 15
        me.screenElements['Altitude-Trend-Indicator']
            .setScale(1,Sy)
            .setTranslation(0, -284.5 * (Sy - 1));
        me._last_alt_ft = alt;
        me._last_alt_s = now;
        settimer(func me.updateALT(), 0.2);
    },
    _last_alt_ft : 0,
    _last_alt_s  : systime(),
#}}}

    updateBARO : func () {
# update BARO widget {{{
        var fmt = me._baro_unit == 'inhg' ? '%.2f%s' : '%i%s';
        me.screenElements['BARO-text']
            .setText(sprintf(fmt,
                        getprop('/instrumentation/altimeter/setting-' ~ me._baro_unit),
                        me._baro_unit == 'inhg' ? 'in' : 'hPa')
                );
    },
    _baro_unit : 'inhg',
#}}}

    updateHSI : func () {
# rotates the compass (PFD) {{{
        var hdg = data.hdg;
        me.screenElements.Rose
            .setRotation(-hdg * D2R);
        me.screenElements['HDG-text']
            .setText(sprintf("%03u°", hdg));
        settimer(func me.updateHSI(), 0.1);
    },
#}}}

    updateHDG : func () {
# moves the heading bug and display heading-deg for 3 seconds (PFD) {{{
        if (me.device.role == 'MFD')
            return;
        var hdg = getprop('/instrumentation/zkv1000/afcs/heading-bug-deg');
        me.screenElements['Heading-bug']
            .setRotation(hdg * D2R);
        me.screenElements['SelectedHDG-bg']
            .show();
        me.screenElements['SelectedHDG-bgtext']
            .show();
        me.screenElements['SelectedHDG-text']
            .setText(sprintf('%03d°%s', hdg, ''))
            .show();
        me.addTimer(3, ['SelectedHDG-text', 'SelectedHDG-bgtext', 'SelectedHDG-bg']);
    },
#}}}

    updateCRS : func () {
# TODO: update display for NAV/GPS/BRG courses {{{
        if (me.device.role == 'MFD')
            return;
        var source = getprop('/instrumentation/zkv1000/cdi/source');
        if (source == 'OFF')
            return;
        var crs = getprop('/instrumentation/zkv1000/cdi/course');
        if (crs == nil)
            return;
        me.screenElements['SelectedCRS-bg']
            .show();
        me.screenElements['SelectedCRS-bgtext']
            .show();
        me.screenElements['SelectedCRS-text']
            .setText(sprintf('%03d°%s', crs, ''))
            .setColor(source == 'GPS' ? me.colors.magenta : me.colors.green)
            .show();
        me.addTimer(3, ['SelectedCRS-text', 'SelectedCRS-bgtext', 'SelectedCRS-bg']);
    },
#}}}

    updateSelectedALT : func {
# animation for altitude section, called via updatedALT {{{
        if (! me.screenElements['SelectedALT'].getVisible())
            return;
        var selected_alt = getprop('/instrumentation/zkv1000/afcs/selected-alt-ft');
        var delta_alt = data.alt - selected_alt;
        if (abs(delta_alt) > 300)
            delta_alt = 300 * abs(delta_alt)/delta_alt;
        me.screenElements['SelectedALT-symbol']
            .setVisible(abs(delta_alt) > 100);
        me.screenElements['SelectedALT-bg']
            .setVisible(abs(delta_alt) > 100);
        me.screenElements['SelectedALT-text']
            .setText(sprintf("%i", selected_alt))
            .setVisible(abs(delta_alt) > 100);
        me.screenElements['SelectedALT-bug']
            .setTranslation(0, delta_alt * 0.567); # 170/300 = 0.567
    },
#}}}

    updateCDI : func {
# animation for CDI {{{
        var source = getprop('/instrumentation/zkv1000/cdi/source');
        if (source == 'OFF') {
            foreach (var s; ['GPS', 'NAV1', 'NAV2'])
                foreach (var t; ['pointer', 'CDI'])
                    me.screenElements[s ~ '-' ~ t].hide();
            me.screenElements['CDI-GPS-ANN-text'].hide();
            me.screenElements['CDI-GPS-XTK-text'].hide();
            me.screenElements['CDI-SRC-text'].hide();
            me.screenElements['CDI'].hide();
        }
        else {
            var course = getprop('/instrumentation/zkv1000/cdi/course');
            var rot = (course - data.hdg) * D2R;
            me.screenElements['CDI']
                .setRotation(rot)
                .show();
            me.screenElements['GPS-CTI']
                .setVisible(getprop('/instrumentation/gps/wp/wp[1]/valid'))
                .setRotation(getprop('/instrumentation/gps/wp/wp[1]/course-deviation-deg') * D2R);
            me.screenElements['GPS-CTI-diamond']
                .setVisible(getprop('/instrumentation/gps/wp/wp[1]/valid'))
                .setRotation(getprop('/instrumentation/gps/wp/wp[1]/course-deviation-deg') * D2R);
            foreach (var s; ['GPS', 'NAV1', 'NAV2']) {
                me.screenElements[s ~ '-pointer']
                    .setRotation(rot)
                    .setVisible(source == s);
                me.screenElements[s ~ '-CDI']
                    .setVisible(source == s);
                foreach (var f; ['FROM', 'TO'])
                    me.screenElements[s ~ '-' ~ f]
                        .setVisible(s == source and getprop('/instrumentation/zkv1000/cdi/' ~ f ~ '-flag'));
                me.screenElements['CDI-SRC-text']
                    .setText(source)
                    .setColor(source == 'GPS' ? me.colors.magenta : me.colors.green)
                    .show();
            }
            var deflection = getprop('/instrumentation/zkv1000/cdi/course-deflection');
            if (left(source, 3) == 'NAV')
                var scale = deflection / 4;
            else { # GPS
                # TODO: deviation depending of the flight phase
                # for now fixed 1 dot = 1 nm course error
                var abs_deflection = abs(deflection);
                me.screenElements['CDI-GPS-XTK-text']
                    .setText(sprintf('XTK %iNM', abs_deflection))
                    .setVisible(abs_deflection > 2.4);
                var scale = deflection / 2;
            }
            scale = (abs(scale) > 1.2) ? 1.2 : scale;
            me.screenElements[source ~ '-CDI']
                .setTranslation(65 * scale, 0);
            settimer(func me.updateCDI(), 0.3);
        }
    },
#}}}

    _updateRadio: func {
# common parts for NAV/LOC/COMM radios{{{
        # arg[0]._r = <comm|nav>
        if (contains(arg[0], "active")) {
            if (arg[0]['active'] == 'none') {
                me.screenElements[arg[0]._r ~ '1-selected-freq']
                    .setColor(1,1,1);
                me.screenElements[arg[0]._r ~ '2-selected-freq']
                    .setColor(1,1,1);
            }
            else {
                me.screenElements[arg[0]._r ~ arg[0]['active'] ~ '-selected-freq']
                    .setColor(0,1,0);
                me.screenElements[arg[0]._r ~ arg[0].inactive ~ '-selected-freq']
                    .setColor(1,1,1);
            }
        }
        if (contains(arg[0], 'tune')) {
            # n = 0 -> NAV1/COMM1
            # n = 1 -> NAV1/COMM2
            me.screenElements[arg[0]._r ~ '-freq-switch']
                .setTranslation(0, arg[0].tune * 25);
            me.screenElements[arg[0]._r ~ (arg[0].tune + 1) ~ '-standby-freq']
                .setColor(0,1,1);
            me.screenElements[arg[0]._r ~ ((arg[0].tune == 0) + 1) ~ '-standby-freq']
                .setColor(1,1,1);
        }
        if (contains(arg[0], 'refresh')) {
            # refresh only one line: NAV1/COMM1 or NAV2/COMM2
            var fmt = (arg[0]._r == 'nav') ? '%.2f' : '%.3f';
            me.screenElements[arg[0]._r ~ arg[0].refresh ~ '-selected-freq']
                .setText(sprintf(fmt, getprop('/instrumentation/'
                               ~ arg[0]._r ~ '[' ~ (arg[0].refresh - 1) ~ ']/frequencies/selected-mhz')));
            me.screenElements[arg[0]._r ~ arg[0].refresh ~ '-standby-freq']
                .setText(sprintf(fmt, getprop('/instrumentation/'
                               ~ arg[0]._r ~ '[' ~ (arg[0].refresh - 1) ~ ']/frequencies/standby-mhz')));
        }
        if (contains(arg[0], 'set')) {
            # positionne la valeur modifiée, les listeners "trigguent" en permanence ces propriétés, donc exit
            var n = getprop('/instrumentation/zkv1000/radios/' ~ arg[0]._r ~ '-tune');
            var fmt = (arg[0]._r == 'nav') ? '%.2f' : '%.3f';
            me.screenElements[arg[0]._r ~ (n + 1) ~ '-standby-freq']
                .setText(sprintf(fmt, getprop('/instrumentation/' ~ arg[0]._r ~ '[' ~ n ~ ']/frequencies/standby-mhz')));
        }
        if (contains(arg[0], 'auto')) {
            # pour rafraichir automagiquement, toutes les deux secondes un refresh pour un NAV
            var radio = arg[0].auto;
            me._updateRadio({refresh: 1, _r: radio});
            settimer(func me._updateRadio({refresh: 2, _r: radio}), 1);
            settimer(func me._updateRadio({auto: radio}), 2);
        }
    },
#}}}

    updateNAV : func {
# update NAV/LOC rodios display upper left (PFD/MFD){{{
        # made active via menu
        if (contains(arg[0], "active")) {
            arg[0]._r = 'nav';
            if (arg[0]['active'] == 'none') {
                me._updateRadio(arg[0]);
                me.screenElements['nav1-id']
                    .setColor(1,1,1);
                me.screenElements['nav2-id']
                    .setColor(1,1,1);
                me.screenElements['NAV1-pointer']
                    .hide();
                me.screenElements['NAV2-pointer']
                    .hide();
            }
            else {
                arg[0].inactive = (arg[0]['active'] == 1) + 1;
                me._updateRadio(arg[0]);
                me.screenElements['nav' ~ arg[0]['active'] ~ '-id']
                    .setColor(0,1,0);
                me.screenElements['NAV' ~ arg[0]['active'] ~ '-pointer']
                    .show();
                me.screenElements['nav' ~ arg[0].inactive ~ '-id']
                    .setColor(1,1,1);
#                me.screenElements['HDI']
#                    .setRotation();
#                me.screenElements['NAV' ~ inactive ~ '-pointer']
#                    .hide();
#                foreach (var e; [ 'FROM', 'TO', 'CDI' ])
#                    me.screenElements['NAV' ~ inactive ~ '-' ~ e]
#                        .hide();
            }
        }
        elsif (contains(arg[0], 'nav-id')) {
            # TODO: récupérer la valeur via les paramètres transmis du listener
            if (arg[0].val == nil)
                arg[0].val = '';
            me.screenElements["nav" ~ arg[0]['nav-id'] ~ "-id"]
                    .setText(arg[0].val);
        }
        else {
            arg[0]._r = 'nav';
            me._updateRadio(arg[0]);
        }
    },
#}}}

    updateCOMM: func {
# update COMM radios display upper right (PFD/MFD){{{
        arg[0]._r = 'comm';
        me._updateRadio(arg[0]);
    },
#}}}

    updateTIME : func {
# updates the displayed time botoom left {{{
        me.screenElements['TIME-text']
            .setText(getprop('/sim/time/gmt-string'));
        settimer(func me.updateTIME(), 1);
    },
#}}}

    updateXPDR : func {
# updates transponder display {{{
        for (var d = 0; d < 4; d+=1)
            me.screenElements['XPDR-DIGIT-' ~ d ~ '-text']
                .setText(sprintf('%i', getprop('/instrumentation/transponder/inputs/digit[' ~ d ~ ']')));
        var tuning = getprop('/instrumentation/zkv1000/radios/xpdr-tuning-digit');
        var fms = getprop('/instrumentation/zkv1000/radios/xpdr-tuning-fms-method');
        for (var d = 0; d < 4; d+=1)
            me.screenElements['XPDR-DIGIT-' ~ d ~ '-text']
                .setColor(1,1,1);
        if (tuning != nil) {
            me.screenElements['XPDR-DIGIT-' ~ tuning ~ '-text']
                .setColor(0,1,1);
            if (fms)
                me.screenElements['XPDR-DIGIT-' ~ (tuning - 1) ~ '-text']
                    .setColor(0,1,1);
        }
        else {
            if (getprop('/instrumentation/transponder/ident'))
                var mode = 'IDENT';
            else
                var mode = getprop('/instrumentation/zkv1000/radio/xpdr-mode');
            var wow = getprop('/gear/gear/wow');
            if (! wow and mode != 'STBY')
                var color = [0, 1, 0];
            else
                var color = [1, 1, 1];
            for (var d = 0; d < 4; d+=1)
                me.screenElements['XPDR-DIGIT-' ~ d ~ '-text']
                    .setColor(color);
            me.screenElements['XPDR-MODE-text']
                .setColor(color)
                .setText(mode);
        }
    },
#}}}

    updateOAT : func {
# update OAT display on normal and reversionnary modes (every 3s) {{{
        var tmp = getprop('/environment/temperature-deg' ~ me._oat_unit);
        me.screenElements['OAT-text']
            .setText(sprintf((abs(tmp) < 10) ? "%.1f %s" : "%i %s", tmp, (me._oat_unit == 'c') ? '°C' : 'F'));
        settimer(func me.updateOAT(), 3);
    },
    _oat_unit : 'c',
#}}}

    updateWindData : func {
# update the window text and arrows for OPTN1/2 {{{
        if (me._winddata_optn == 0)
            return;
        if (data.ias < 30) {
            me.screenElements['WindData-NODATA']
                .hide();
            var wind_hdg = getprop('/environment/wind-from-heading-deg');
            var wind_spd = getprop('/environment/wind-speed-kt');
            var alpha = wind_hdg - data.hdg;
            if (me._winddata_optn == 1) {
                me.screenElements['WindData-OPTN1-HDG']
                    .setRotation((alpha + 180) * D2R)
                    .show();
                me.screenElements['WindData-OPTN1-HDG-text']
                    .setText(sprintf("%03i°", wind_hdg))
                    .show();
                me.screenElements['WindData-OPTN1-SPD-text']
                    .setText(int(wind_spd) ~ 'KT')
                    .show();
            }
            else { # me._winddata_optn == 2
                alpha *= D2R;
                var Vt = wind_spd * math.sin(alpha);
                var Ve = wind_spd * math.cos(alpha);
                if (Vt != 0) {
                    me.screenElements['WindData-OPTN2-crosswind-text']
                        .setText(sprintf('%i', abs(Vt)))
                        .show();
                    me.screenElements['WindData-OPTN2-crosswind']
                        .setScale(-abs(Vt)/Vt, 1)
                        .setTranslation(-35 * (abs(Vt)/Vt + 1), 0)
                        .show();
                }
                if (Ve != 0) {
                    me.screenElements['WindData-OPTN2-headwind-text']
                        .setText(sprintf('%i', abs(Ve)))
                        .show();
                    me.screenElements['WindData-OPTN2-headwind']
                        .setScale(1, abs(Ve)/Ve)
                        .setTranslation(0, 515 * (1 - abs(Ve)/Ve))
                        .show();
                }
            }
        }
        else {
            foreach (var e; [
                    'WindData-OPTN1-HDG',
                    'WindData-OPTN1-HDG-text',
                    'WindData-OPTN1-SPD-text',
                    'WindData-OPTN2-crosswind-text',
                    'WindData-OPTN2-crosswind',
                    'WindData-OPTN2-headwind-text',
                    'WindData-OPTN2-headwind'
            ])
                me.screenElements[e].hide();
            me.screenElements['WindData-NODATA'].show();
        }
        settimer(func me.updateWindData(), 0.5);
    },
    _winddata_optn : 0,
#}}}

    updateAOA : func {
# update Angle Of Attack {{{
        if (me.device.data.aoa == 0)
            return;
        var color = [1,1,1];
        var norm = data.aoa / data['stall-aoa'];
        me.screenElements['AOA-text']
            .setText(sprintf('% .1f', norm));
        if (norm > 1) norm = 1;
        if (norm > 0.9)
            color = [1,0,0];
        elsif (norm > 0.7)
            color = [1,1,0];
        elsif (norm < 0) {
            norm = 0;
            color = [1,0,0];
        }
        me.screenElements['AOA-needle']
            .setRotation(-norm * math.pi)
            .setColorFill(color);
        me.screenElements['AOA-text']
            .setColor(color);
        settimer(func me.updateAOA(), 0.1);
    },
# }}}

    updateBRG : func {
# displays and update BRG1/2 {{{
        foreach (var brg; [1, 2]) {
            var source = 'brg' ~ brg ~ '-source';

            var dev = radios.getNode(source).getValue();
            var el  = 'BRG' ~ brg;
            if (dev != 'OFF') {
                var info = {
                    pointer : nil,
                    id : 'NO DATA',
                    hdg : nil,
                    dst : '--.-NM'
                };
                if (left(dev, 3) == 'NAV') {
                    info.pointer = getprop('/instrumentation/nav[' ~ (brg - 1) ~ ']/in-range');
                    if (info.pointer) {
                        info.id  = getprop('/instrumentation/nav[' ~ (brg - 1) ~ ']/nav-id');
                        info.hdg = getprop('/instrumentation/nav[' ~ (brg - 1) ~ ']/heading-deg');
                        info.dst = sprintf('%.1d', getprop('/instrumentation/nav[' ~ (brg - 1) ~ ']/nav-distance') / 1852); # m -> /1852
                    }
                }
                elsif (dev == 'GPS') {
                    info.pointer = props.getNode('/instrumentation/gps/wp').getChild('wp[1])');
                    if (info.pointer) {
                        info.id  = getprop('/instrumentation/gps/wp/wp[1]/ID');
                        info.hdg = getprop('/instrumentation/gps/wp/wp[1]/bearing-mag-deg');
                        info.dst = sprintf('%.1d', getprop('/instrumentation/gps/wp/wp[1]/distance-nm'));
                    }
                }
                else { # there are 2 available ADF in FG, but instrument manage only 1
                    info.pointer = getprop('/instrumentation/adf/in-range');
                    if (info.pointer) {
                        info.id  = getprop('/instrumentation/adf/ident');
                        info.hdg = getprop('/instrumentation/adf/indicated-bearing-deg');
                    }
                }

                if (info.pointer)
                    me.screenElements[el ~ '-pointer']
                        .setRotation(-info.hdg - data.hdg * D2R)
                        .show();
                else
                    me.screenElements[el ~ '-pointer']
                        .hide();
                me.screenElements[el ~ '-SRC-text']
                    .setText(dev);
                me.screenElements[el ~ '-DST-text']
                    .setText(info.dst);
                me.screenElements[el ~ '-WPID-text']
                    .setText(info.id);
                me.screenElements['BRG' ~ brg]
                    .show();
            }
            else {
                me.screenElements['BRG' ~ brg]
                    .hide();
            }
        }
        settimer(func me.updateBRG(), 0.5);
    },
#}}}

    updateOMI : func {
# display marker baecon Outer, Middle, Inner {{{
        var marker = nil;
        foreach (var m; ['outer', 'middle', 'inner'])
            if (getprop('/instrumentation/marker-beacon/' ~ m)) {
                marker = m;
                me.screenElements['OMI']
                    .show();
                break;
            }
        if (marker != nil) {
            me.screenElements['MarkerText']
                .setText(me._omi_data[marker].t)
                .show();
            me.screenElements['MarkerBG']
                .setColorFill(me._omi_data[marker].bg)
                .show();
        }
        else
            me.screenElements['OMI']
                .hide();
        settimer(func me.updateOMI(), 1);
    },
    _omi_data : {
        'outer':  {t: 'O', bg: [0,1,1]},
        'middle': {t: 'M', bg: [1,1,1]},
        'inner':  {t: 'I', bg: [1,1,0]},
    },
#}}}
};
