# Simulation of Kanardia HORIS by Benedikt Wolf (D-ECHO) based on

# A3XX Lower ECAM Canvas
# Joshua Davidson (Octal450)

#	References:
#		https://www.kanardia.eu/wp-content/uploads/2019/01/HorisManual.pdf

var HORIS_adahrs = nil;
var HORIS_display = nil;

var instrument_prop = props.globals.initNode("instrumentation/horis");

var AI_pitch = props.globals.getNode("instrumentation/horis-ahrs/pitch-deg", 1);
var AI_roll = props.globals.getNode("instrumentation/horis-ahrs/roll-deg", 1);

var ASI_ias = props.globals.getNode("instrumentation/horis-airspeed-indicator/indicated-speed-kt", 1);
var ASI_tas = props.globals.getNode("instrumentation/horis-airspeed-indicator/true-speed-kt", 1);

var ALT = props.globals.getNode("instrumentation/horis-altimeter/indicated-altitude-ft", 1);
var ALT_qnh = props.globals.getNode("instrumentation/horis-altimeter/setting-hpa", 1);

var WIND_hdg = props.globals.getNode("environment/wind-from-heading-deg", 1);
var WIND_speed = props.globals.getNode("environment/wind-speed-kt", 1);

var HDG = props.globals.getNode("orientation/heading-deg", 1);

var OAT_degc = props.globals.getNode("environment/temperature-degc", 1);

var SLIPSKID = props.globals.getNode("instrumentation/slip-skid-ball/indicated-slip-skid", 1);

var TURN = props.globals.getNode("instrumentation/horis-ahrs/indicated-turn-rate-degps", 1);

var VS = props.globals.getNode("instrumentation/vertical-speed-indicator/indicated-speed-fpm", 1);

var airspeed_limits = {
	vs0:	getprop("instrumentation/horis/limits/vs0"),
	vs1:	getprop("instrumentation/horis/limits/vs1"),
	vfe:	getprop("instrumentation/horis/limits/vfe"),
	va:		getprop("instrumentation/horis/limits/va"),
	vne:	getprop("instrumentation/horis/limits/vne"),
};

var timeout = 0.0;

var volts = props.globals.getNode( "systems/electrical/outputs/horis", 1 );

var instrument_dir = "Aircraft/Phoenix/Models/Instruments/HORIS/";

var magu_available = ( getprop("instrumentation/horis/magu-available") != nil and getprop("instrumentation/horis/magu-available") == 1 );

var gps = {
	gs:	props.globals.getNode("instrumentation/horis-gps/indicated-ground-speed-kt", 1),
	track:	[
		props.globals.getNode("instrumentation/horis-gps/indicated-track-true-deg", 1),
		props.globals.getNode("instrumentation/horis-gps/indicated-track-magnetic-deg", 1),
	],
};
# 
var time = props.globals.getNode("/sim/time/elapsed-sec", 1);

var brightness = instrument_prop.initNode("brightness", 1.0, "DOUBLE");

var menu = {
	main: {
		show: 0,	# main menu, 0 = off, 1 = on
		show_p: instrument_prop.initNode("main-menu", 0, "BOOL"),
		active_sel: instrument_prop.initNode("main-menu/active-sel", 0, "INT"),	# 0/1/2 selected (top to bottom)
		main_active_sel_state: -1,
		set_brightness: instrument_prop.initNode("main-menu/set-brightness", 0, "BOOL"),
		set_brightness_state: -1,
	},
	settings: {
		show: 0,
		show_p: instrument_prop.initNode("settings-menu", 0, "BOOL"),
		page: -1,
		active_sel: instrument_prop.initNode("settings-menu/active-sel", 0, "INT"),	# 0 - 11, page selection done in code
		active_sel_state: -1,
		labels: [ "Units", "Airspeed", "Response", "AHRS Level", "Screens", "Direction Ind.", "G-Meter", "Pitostatic Offset", "Turn Rate", "Time Zone", "NMEA Out", "Init. Brightness", "Security", "GNSS Info", "About" ],
	},
	units: {
		show: 0,	# units menu, 0 = off, 1 = on
		show_p: instrument_prop.initNode("units-menu", 0, "BOOL"),
		page: -1,
		active_sel: instrument_prop.initNode("units-menu/active-sel", 0, "INT"),	# 0 - 6, page selection done in code
		active_sel_state: -1,
		active_change: instrument_prop.initNode("units-menu/active-change", 0, "INT"),	# 0 - 7, 0 = off, 1 - 6
		active_change_state: -1,
		labels: [ "Direction", "Speed", "Vario", "Wind", "Altitude", "QNH", "Temp." ],
		values: [
			[ "True", "Magnetic" ],
			[ "kts", "km/h", "mph" ],
			[ "m/s", "ft/min", "kts" ],
			[ "m/s", "ft/min", "kts" ],
			[ "meter", "feet" ],
			[ "hPa", "inHg" ],
			[ "° C", "° F" ],
		],
		properties: [
			instrument_prop.getNode("units-menu/direction"),
			instrument_prop.getNode("units-menu/speed"),
			instrument_prop.getNode("units-menu/vario"),
			instrument_prop.getNode("units-menu/wind"),
			instrument_prop.getNode("units-menu/altitude"),
			instrument_prop.getNode("units-menu/qnh"),
			instrument_prop.getNode("units-menu/temperature"),
		],
	},
};

var units_val = [ 0, 0, 0, 0, 0, 0, 0 ];

var azimuth_ref = [ "T", "M" ];

var coordinates = {
	main_menu_timeout: nil,
	settings_menu_timeout: nil,
	units_menu_timeout: nil,
	vs_bar: nil,
	turnrate_bar: nil,
	ias_tape: nil,
};

#roundToNearest function used for alt tape, thanks @Soitanen (737-800)!
var roundToNearest = func(n, m) {
	var x = int(n/m)*m;
	if((math.mod(n,m)) > (m/2) and n > 0)
			x = x + m;
	if((m - (math.mod(n,m))) > (m/2) and n < 0)
			x = x - m;
	return x;
}

var canvas_HORIS_base = {
	init: func(canvas_group, file) {
		canvas.parsesvg(canvas_group, file, {'font-mapper': global.canvas.FontMapper});

		var svg_keys = me.getKeys();
		foreach (var key; svg_keys) {
			me[key] = canvas_group.getElementById(key);
		}
		
		if( !magu_available ){
			me["wind.digits"].hide();
			me["wind.arrow"].hide();
		}
		
		me.h_trans = me["horizon"].createTransform();
		me.h_rot = me["horizon"].createTransform();
		
		coordinates.main_menu_timeout = [ me["main_menu.timeout_bar"].get( "coord[0]" ), me["main_menu.timeout_bar"].get( "coord[2]" ) ];
		coordinates.settings_menu_timeout = [ me["settings_menu.timeout_bar"].get( "coord[0]" ), me["settings_menu.timeout_bar"].get( "coord[2]" ) ];
		coordinates.units_menu_timeout = [ me["units_menu.timeout_bar"].get( "coord[0]" ), me["units_menu.timeout_bar"].get( "coord[2]" ) ];
		
		coordinates.vs_bar = [ me["vs.bar"].get( "coord[1]" ), me["vs.bar"].get( "coord[3]" ) ];
		coordinates.turnrate_bar = [ me["turnrate.bar"].get( "coord[0]" ), me["turnrate.bar"].get( "coord[2]" ) ];

		coordinates.ias_tape = me["white_bar"].get( "coord[3]" );

		me.page = canvas_group;

		return me;
	},
	getKeys: func() {
		return [];
	},
	update: func() {
		if ( volts.getDoubleValue() > 10 ) {
			HORIS_adahrs.page.show();
			HORIS_adahrs.update();
		} else {
			HORIS_adahrs.page.hide();
		}
	},
};

var pitch = nil;
var roll = nil;
var ias = nil;
var tas = nil;
var altitude = nil;
var qnh = nil;
var wind_from = nil;
var wind_spd = nil;
var vs_val = nil;
var azimuth_active = -1;
var gs = nil;
var last_alt_tape_state = -1;
var altNumLow = nil;
var altNumHigh = nil;
var altNumCenter = nil;
var last_qnh = "";

var canvas_HORIS_adahrs = {
	new: func(canvas_group, file) {
		var m = { parents: [canvas_HORIS_adahrs , canvas_HORIS_base] };
		m.init(canvas_group, file);

		return m;
	},
	getKeys: func() {
		return ["white_bar", "yellow_bar", "green_bar", "red_bar",
		"horizon","rollpointer",
		"ias.digits","tas.digits","asi.tape",
		"alt.digits.small","alt.digits.big",
		"qnh.digits",
		"wind.digits","wind.arrow",
		"azimuth.digits","azimuth.source","azimuth.reference","azimuth.cross",
		"gs.digits",
		"oat.digits", "oat.unit",
		"ball",
		"turnrate.bar",
		"vs.digits","vs.bar",
		"altTapeScale","altTextHigh1","altTextHigh2","altTextHigh3","altTextHigh4","altTextHigh5","altTextHigh6","altTextHigh7","altTextHigh8","altTextHigh9",
		"altTextLow1","altTextLow2","altTextLow3","altTextLow4","altTextLow5","altTextLow6","altTextLow7","altTextLow8",
		"altTextHighSmall2","altTextHighSmall3","altTextHighSmall4","altTextHighSmall5","altTextHighSmall6","altTextHighSmall7","altTextHighSmall8","altTextHighSmall9",
		"altTextLowSmall1","altTextLowSmall2","altTextLowSmall3","altTextLowSmall4","altTextLowSmall5","altTextLowSmall6","altTextLowSmall7","altTextLowSmall8",
		"main_menu","main_menu.timeout_bar","main_menu.active_selection",
		"main_menu.brightness.bg","main_menu.brightness.digits",
		"settings_menu","settings_menu.timeout_bar","settings_menu.active_selection",
		"settings_menu.label1", "settings_menu.label2", "settings_menu.label3", "settings_menu.label4", "settings_menu.label5", "settings_menu.scroll_bar",
		"settings_menu.icons1", "settings_menu.icons2", "settings_menu.icons3",
		"units_menu", "units_menu.timeout_bar", "units_menu.active_selection", "units_menu.scroll_bar",
		"units_menu.label1", "units_menu.label2", "units_menu.label3", "units_menu.label4", "units_menu.label5", 
		"units_menu.value1", "units_menu.value2", "units_menu.value3", "units_menu.value4", "units_menu.value5", 
		"units_menu.value1.bg", "units_menu.value2.bg", "units_menu.value3.bg", "units_menu.value4.bg", "units_menu.value5.bg", 
		];
	},
	update: func() {
		
		#Attitude Indicator
		pitch = AI_pitch.getValue();
		roll =  AI_roll.getValue();
		
		me.h_trans.setTranslation(0,pitch*3.36);
		me.h_rot.setRotation(-roll*D2R,me["horizon"].getCenter());
		
		if(roll<45 and roll>-45){
			me["rollpointer"].setRotation(-roll*D2R);
		}
		
		#Airspeed
		ias = ASI_ias.getDoubleValue();
		tas = ASI_tas.getDoubleValue();
		if( units_val[1] == 1 ){
			ias *= KT2MPS * 3.6;
			tas *= KT2MPS * 3.6;
		} elsif( units_val[1] == 2 ){
			ias *= 1.15078; # kts to mph
			tas *= 1.15078; # kts to mph
		}
		
		me["ias.digits"].setText(sprintf("%3d", math.round(ias)));
		me["asi.tape"].setTranslation(0,ias*2.95);
		me["tas.digits"].setText(sprintf("%3d", math.round(tas)));
		
		#Altitude
		altitude = ALT.getDoubleValue();
		if( units_val[4] == 0 ){
			altitude *= FT2M;
		}
		
		me["alt.digits.big"].setText(sprintf("%3d", math.floor( math.round( altitude, 10 ) /100, 1) ) );
		me["alt.digits.small"].setText(sprintf("%02d", 100 * math.mod( math.round( altitude, 10 ) / 100, 1 ) ) );
		
		#Alt Tape
		me["altTapeScale"].setTranslation( 0, (altitude - roundToNearest(altitude, 1000) ) * 0.293 );
		
		var last_altNumLow = altNumLow;
		
		if (roundToNearest(altitude, 1000) == 0 ) {
			if( last_alt_tape_state != 0 ){
				last_alt_tape_state = 0;
				me["altTextLowSmall1"].setText("100");
				me["altTextLowSmall2"].setText("200");
				me["altTextLowSmall3"].setText("300");
				me["altTextLowSmall4"].setText("400");
				me["altTextLowSmall5"].setText("500");
				me["altTextLowSmall6"].setText("600");
				me["altTextLowSmall7"].setText("700");
				me["altTextLowSmall8"].setText("800");
				me["altTextHighSmall2"].setText("100");
				me["altTextHighSmall3"].setText("200");
				me["altTextHighSmall4"].setText("300");
				me["altTextHighSmall5"].setText("400");
				me["altTextHighSmall6"].setText("500");
				me["altTextHighSmall7"].setText("600");
				me["altTextHighSmall8"].setText("700");
				me["altTextHighSmall9"].setText("800");
				altNumLow = "-";
				altNumHigh = "";
				altNumCenter = altNumHigh;
			}
		} elsif (roundToNearest(altitude, 1000) > 0) {
			if( last_alt_tape_state != 1 ){
				last_alt_tape_state = 1;
				me["altTextLowSmall1"].setText("900");
				me["altTextLowSmall2"].setText("800");
				me["altTextLowSmall3"].setText("700");
				me["altTextLowSmall4"].setText("600");
				me["altTextLowSmall5"].setText("500");
				me["altTextLowSmall6"].setText("400");
				me["altTextLowSmall7"].setText("300");
				me["altTextLowSmall8"].setText("200");
				me["altTextHighSmall2"].setText("100");
				me["altTextHighSmall3"].setText("200");
				me["altTextHighSmall4"].setText("300");
				me["altTextHighSmall5"].setText("400");
				me["altTextHighSmall6"].setText("500");
				me["altTextHighSmall7"].setText("600");
				me["altTextHighSmall8"].setText("700");
				me["altTextHighSmall9"].setText("800");
			}
			altNumLow = roundToNearest(altitude, 1000)/1000 - 1;
			altNumHigh = roundToNearest(altitude, 1000)/1000;
			altNumCenter = altNumHigh;
		} elsif (roundToNearest(altitude, 1000) < 0) {
			if( last_alt_tape_state != 2 ){
				last_alt_tape_state = 2;
				me["altTextLowSmall1"].setText("100");
				me["altTextLowSmall2"].setText("200");
				me["altTextLowSmall3"].setText("300");
				me["altTextLowSmall4"].setText("400");
				me["altTextLowSmall5"].setText("500");
				me["altTextLowSmall6"].setText("600");
				me["altTextLowSmall7"].setText("700");
				me["altTextLowSmall8"].setText("800");
				me["altTextHighSmall2"].setText("900");
				me["altTextHighSmall3"].setText("800");
				me["altTextHighSmall4"].setText("700");
				me["altTextHighSmall5"].setText("600");
				me["altTextHighSmall6"].setText("500");
				me["altTextHighSmall7"].setText("400");
				me["altTextHighSmall8"].setText("300");
				me["altTextHighSmall9"].setText("200");
			}
			altNumLow = roundToNearest(altitude, 1000)/1000;
			altNumHigh = roundToNearest(altitude, 1000)/1000 + 1;
			altNumCenter = altNumLow;
		}
		if ( altNumLow == 0 ) {
			altNumLow = "";
		}
		if ( altNumHigh == 0 and altitude < 0) {
			altNumHigh = "-";
		}
		if( altNumLow != last_altNumLow ){
			me["altTextLow1"].setText(sprintf("%s", altNumLow));
			me["altTextLow2"].setText(sprintf("%s", altNumLow));
			me["altTextLow3"].setText(sprintf("%s", altNumLow));
			me["altTextLow4"].setText(sprintf("%s", altNumLow));
			me["altTextLow5"].setText(sprintf("%s", altNumLow));
			me["altTextLow6"].setText(sprintf("%s", altNumLow));
			me["altTextLow7"].setText(sprintf("%s", altNumLow));
			me["altTextLow8"].setText(sprintf("%s", altNumLow));
			me["altTextHigh1"].setText(sprintf("%s", altNumCenter));
			me["altTextHigh2"].setText(sprintf("%s", altNumHigh));
			me["altTextHigh3"].setText(sprintf("%s", altNumHigh));
			me["altTextHigh4"].setText(sprintf("%s", altNumHigh));
			me["altTextHigh5"].setText(sprintf("%s", altNumHigh));
			me["altTextHigh6"].setText(sprintf("%s", altNumHigh));
			me["altTextHigh7"].setText(sprintf("%s", altNumHigh));
			me["altTextHigh8"].setText(sprintf("%s", altNumHigh));
			me["altTextHigh9"].setText(sprintf("%s", altNumHigh));
		}
		
		# QNH
		qnh = ALT_qnh.getDoubleValue();
		if( units_val[5] == 0 ){
			qnh = sprintf("%4d", math.round( qnh ) );
		}elsif( units_val[5] == 1 ){
			qnh = sprintf("%4.2f", qnh * 0.02953 ); # hPa to inHg
		}
		if( qnh != last_qnh ){
			me["qnh.digits"].setText(qnh);
			last_qnh = qnh;
		}
		
		# Azimuth and Groundspeed, calculated from GPS
		gs = gps.gs.getDoubleValue();;
		if( units_val[1] == 1 ){
			gs *= KT2MPS / 3.6;
		} elsif( units_val[1] == 2 ){
			gs *= 1.15078; # kts to mph
		}
		if( gs > 1 ){
			if( azimuth_active != 1 ){
				azimuth_active = 1;
				me["azimuth.cross"].hide();
			}
			me["azimuth.digits"].setText( sprintf("%3d", math.round( gps.track[ units_val[0] ].getDoubleValue() ) ) );
		} elsif( azimuth_active != 0 ){
			azimuth_active = 0;
			me["azimuth.cross"].show();
			me["azimuth.digits"].setText("---");
		}
		me["gs.digits"].setText( sprintf("%3d", math.round( gs ) ) );
		
		#Wind Indication
		if( magu_available ){
			wind_from = WIND_hdg.getValue();
			wind_spd = WIND_speed.getValue(); # in kts
			if( units_val[3] == 0 ){
				wind_spd *= KT2MPS;
			} elsif( units_val[3] == 1 ){
				wind_spd *= KT2FPS / 60;
			}
		
			me["wind.digits"].setText(sprintf("%2d", math.round(wind_spd))~"/"~sprintf("%3d", math.round(wind_from))~"°");
			me["wind.arrow"].setRotation( ( heading -  wind_from ) * -D2R);
		}
		
		#OAT
		var oat = OAT_degc.getDoubleValue();
		if( units_val[6] == 1 ) {
			oat = oat * 1.8 + 32; # degree Fahrenheit
		}
		me["oat.digits"].setText(sprintf("%2d", math.round( oat )));
		
		#Ball/Slip-Skid
		me["ball"].setTranslation(SLIPSKID.getDoubleValue()*-25, 0);
		
		# Turn Rate
		# the coordinates show one standard deflection to the right
		me["turnrate.bar"].set( "coord[2]", coordinates.turnrate_bar[0] + ( coordinates.turnrate_bar[1] - coordinates.turnrate_bar[0] ) * math.clamp( TURN.getDoubleValue() / 6, -1, 1 ) );
		
		#Vertical Speed
		vs_val = VS.getDoubleValue(); # in fpm
		
		me["vs.bar"].set( "coord[3]", coordinates.vs_bar[0] + ( ( coordinates.vs_bar[1] - coordinates.vs_bar[0] ) * math.clamp( vs_val / 2000, -1, 1 ) ) );
		
		if( units_val[2] == 0 ){
			vs_val *= FT2M / 60;
		} elsif( units_val[2] == 1 ){
			vs_val *= 0.001;
		} elsif( units_val[2] == 2 ){
			vs_val *= 60 / KT2FPS;
		}
		me["vs.digits"].setText( sprintf("%3.1f", vs_val ) );
		
		# Main Menu
		if( menu.main.show ){
			me["main_menu.timeout_bar"].set( "coord[2]", coordinates.main_menu_timeout[0] + ( ( coordinates.main_menu_timeout[1] - coordinates.main_menu_timeout[0] ) * ( 1 - timeout ) ) );
		}
		
		# Settings Menu
		if( menu.settings.show ){
			me["settings_menu.timeout_bar"].set( "coord[2]", coordinates.settings_menu_timeout[0] + ( ( coordinates.settings_menu_timeout[1] - coordinates.settings_menu_timeout[0] ) * ( 1 - timeout ) ) );
		}
		
		# Units Menu
		if( menu.units.show ){
			me["units_menu.timeout_bar"].set( "coord[2]", coordinates.units_menu_timeout[0] + ( ( coordinates.units_menu_timeout[1] - coordinates.units_menu_timeout[0] ) * ( 1 - timeout ) ) );
		}
	},
	# _show functions: _show(0): hide; _show(1): show
	main_menu_show: func( i ){
		menu.main.show = i;
		menu.main.show_p.setBoolValue( i );
		if( i ){
			me["main_menu"].show();
			me.main_on_active_sel();
			me.main_on_brightness_state();
		} else {
			me["main_menu"].hide();
		}
	},
	main_on_active_sel: func{
		menu.main.active_sel_state = menu.main.active_sel.getIntValue();
		me["main_menu.active_selection"].setTranslation( 0, menu.main.active_sel_state * 37.5 );
	},
	main_on_brightness_state: func {
		menu.main.set_brightness_state = menu.main.set_brightness.getBoolValue();
		if( menu.main.set_brightness_state ){
			me["main_menu.brightness.bg"].show();
			me["main_menu.brightness.digits"].setColor( 0, 0, 0 );
			me["main_menu.timeout_bar"].hide();
		} else {
			me["main_menu.brightness.bg"].hide();
			me["main_menu.brightness.digits"].setColor( 1, 1, 1 );
			me["main_menu.timeout_bar"].show();
		}
	},
	main_on_brightness_change: func {
		me["main_menu.brightness.digits"].setText( sprintf("%3d", math.round( brightness.getDoubleValue() * 100 ) ) ~ "%" );
	},
	settings_menu_show: func( i ){
		menu.settings.show = i;
		menu.settings.show_p.setBoolValue( i );
		if( i ){
			me["settings_menu"].show();
			me.settings_on_active_sel();
		} else {
			me["settings_menu"].hide();
		}
	},
	settings_on_active_sel: func {
		menu.settings.active_sel_state = menu.settings.active_sel.getIntValue();
		if( int( menu.settings.active_sel_state / 5 ) != menu.settings.page ){
			menu.settings.page = int( menu.settings.active_sel_state / 5 );
			me["settings_menu.scroll_bar"].setTranslation( 0, menu.settings.page * 63.254 );
			me["settings_menu.label1"].setText( menu.settings.labels[ 5 * menu.settings.page ] );
			me["settings_menu.label2"].setText( menu.settings.labels[ 5 * menu.settings.page + 1 ] );
			me["settings_menu.label3"].setText( menu.settings.labels[ 5 * menu.settings.page + 2 ] );
			me["settings_menu.label4"].setText( menu.settings.labels[ 5 * menu.settings.page + 3 ] );
			me["settings_menu.label5"].setText( menu.settings.labels[ 5 * menu.settings.page + 4 ] );
			if( menu.settings.page == 0 ){
				me["settings_menu.icons1"].show();
				me["settings_menu.icons2"].hide();
				me["settings_menu.icons3"].hide();
			} elsif( menu.settings.page == 1 ){
				me["settings_menu.icons1"].hide();
				me["settings_menu.icons2"].show();
				me["settings_menu.icons3"].hide();
			} else {
				me["settings_menu.icons1"].hide();
				me["settings_menu.icons2"].hide();
				me["settings_menu.icons3"].show();
			}
		}
		me["settings_menu.active_selection"].setTranslation( 0, ( menu.settings.active_sel_state - 5 * menu.settings.page ) * 37.5 );
	},
	update_unit: func( i ){
		if( menu.units.page >= 0 ) me["units_menu.value"~ ( 1 + i - 2 * menu.units.page ) ].setText( menu.units.values[ i ][ menu.units.properties[ i ].getIntValue() ] );
		units_val[ i ] = menu.units.properties[i].getIntValue();
		if( i == 0 ){
			me["azimuth.reference"].setText( azimuth_ref[ units_val[0] ] );
		} elsif( i == 6 ){
			me["oat.unit"].setText( string.replace( menu.units.values[6][units_val[6]], " ", "" ) );
		}

		var unit_factor = 1.0;
		if( units_val[1] == 1 ){
			unit_factor = KT2MPS * 3.6;
		} elsif( units_val[1] == 2 ){
			unit_factor = 1.15078; # kts to mph
		}

		me["white_bar"].set( "coord[1]", coordinates.ias_tape - airspeed_limits.vs0 * unit_factor * 2.95 );
		me["white_bar"].set( "coord[3]", coordinates.ias_tape - airspeed_limits.vfe * unit_factor * 2.95 );

		me["green_bar"].set( "coord[1]", coordinates.ias_tape - airspeed_limits.vs1 * unit_factor * 2.95 );
		me["green_bar"].set( "coord[3]", coordinates.ias_tape - airspeed_limits.va * unit_factor * 2.95 );

		me["yellow_bar"].set( "coord[1]", coordinates.ias_tape - airspeed_limits.va * unit_factor * 2.95 );
		me["yellow_bar"].set( "coord[3]", coordinates.ias_tape - airspeed_limits.vne * unit_factor * 2.95 );

		me["red_bar"].set( "coord[1]", coordinates.ias_tape - airspeed_limits.vne * unit_factor * 2.95 );
		me["red_bar"].set( "coord[3]", coordinates.ias_tape - 350 * 2.95 );
	},
	units_menu_show: func( i ){
		menu.units.show = i;
		menu.units.show_p.setBoolValue( i );
		if( i ){
			me["units_menu"].show();
			me.units_on_active_sel();
			me.units_on_active_change();
		} else {
			me["units_menu"].hide();
		}
	},
	units_on_active_change: func {
		menu.units.active_change_state = menu.units.active_change.getIntValue();
		for( var i = 1; i <= 5; i += 1 ){
			if( i == ( menu.units.active_change_state - 2 * menu.units.page ) ){
				me["units_menu.value"~i~".bg"].show();
				me["units_menu.value"~i].setColor( 0, 0, 0 );
			} else {
				me["units_menu.value"~i~".bg"].hide();
				me["units_menu.value"~i].setColor( 1, 1, 1 );
			}
		}
	},
	units_on_active_sel: func {
		menu.units.active_sel_state = menu.units.active_sel.getIntValue();
		if( ( menu.units.active_sel_state > 4 ) != menu.units.page ){
			menu.units.page = ( menu.units.active_sel_state > 4 );
			me["units_menu.scroll_bar"].setTranslation( 0, menu.units.page * 56.23 );
			for( var i = 1; i <= 5; i += 1 ){
				me["units_menu.label"~i].setText( menu.units.labels[ 2 * menu.units.page +  i - 1 ] );
				me["units_menu.value"~i].setText( menu.units.values[ 2 * menu.units.page +  i - 1 ][ menu.units.properties[ 2 * menu.units.page + i - 1 ].getIntValue() ] );
			}
		}
		me["units_menu.active_selection"].setTranslation( 0, ( menu.units.active_sel_state - 2 * menu.units.page ) * 37.5 );
	},
};

var base_updater = maketimer( 0.02, canvas_HORIS_base.update );
base_updater.simulatedTime = 1;

setlistener("sim/signals/fdm-initialized", func {
	HORIS_display = canvas.new({
		"name": "HORIS",
		"size": [320, 240],
		"view": [320, 240],
		"mipmapping": 1
	});
	
	HORIS_display.addPlacement({"node": "HORIS.display"});
	
	HORIS_adahrs = canvas_HORIS_adahrs.new( HORIS_display.createGroup(), instrument_dir~"HORIS_ADAHRS.svg");
	
	for( var i = 0; i <= 6; i += 1 ){
		HORIS_adahrs.update_unit( i );
	}
	HORIS_adahrs.main_menu_show( 0 );
	HORIS_adahrs.settings_menu_show( 0 );
	HORIS_adahrs.units_menu_show( 0 );
	
	base_updater.start();
});

var cur_timeout = 0.0;

var timeout_bar = func {
	# https://www.youtube.com/watch?v=ASEE_p26Gl8 ( approx. 7 min, 5 sec ): Timeout time is approx. 6-7 seconds
	if( timeout > 0 and menu.main.set_brightness_state <= 0 and menu.units.active_change_state <= 0 ){
		timeout -= 0.0154;	# Use the timer interval ( divided by 6.5 ) here as we don't need precission
		if( timeout <= 0 ){
			timeout = 0;
			timeout_elapsed();
		}
	}
}

var timeout_elapsed = func {
	if( menu.units.show ){
		HORIS_adahrs.units_menu_show(0);
		menu.units.active_sel.setIntValue( 0 );
		menu.units.active_change.setIntValue( 0 );
		HORIS_adahrs.settings_menu_show(1);
		timeout = 1;
	} else {
		HORIS_adahrs.main_menu_show(0);
		menu.main.active_sel.setIntValue( 0 );;
		HORIS_adahrs.settings_menu_show(0);
		menu.settings.active_sel.setIntValue( 0 );
		timeout_loop.stop();
	}
}

var timeout_loop = maketimer( 0.1, timeout_bar );
timeout_loop.simulatedTime = 1;

var button = {
	new: func( interval, func_short, func_long, condition = func return 1 ){
		var obj = {
			parents: [button],
			interval: interval,
			func_short: func_short,
			func_long: func_long,
			time_pressed: 0.0,
			is_pressed: 0,
			condition: condition,
		};
		return obj;
	},
	pressed: func( a ){
		if( me.condition() ){
			if( a and !me.is_pressed ){
				me.time_pressed = time.getDoubleValue();
				me.is_pressed = 1;
				settimer( func me.pressed(0), me.interval );
				# Reset Timeout
				timeout = 1;
				if( !timeout_loop.isRunning ){
					timeout_loop.start();
				}
			} elsif( !a and me.is_pressed ){
				# Reset Timeout
				timeout = 1;
				if( !timeout_loop.isRunning ){
					timeout_loop.start();
				}
				me.is_pressed = 0;
				if( ( time.getDoubleValue() - me.time_pressed ) >= me.interval ){
					me.func_long();
				} else {
					me.func_short();
				}
				me.time_pressed = 0.0;
			}
		}
	},
};

var main_knob_short = func {
	# confirm current selection
	if( menu.main.show ){
		if( menu.main.active_sel_state == 0 ){
			screen.log.write("HORIS: Pitch Reset Not Yet Implemented");
		} elsif( menu.main.active_sel_state == 1 ){
			if( menu.main.set_brightness_state != 1 ){
				menu.main.set_brightness.setBoolValue( 1 );
				HORIS_adahrs.main_on_brightness_state();
			} else {
				menu.main.set_brightness.setBoolValue( 0 );
				HORIS_adahrs.main_on_brightness_state();
			}
		} else {
			HORIS_adahrs.settings_menu_show( 1 );
			HORIS_adahrs.main_menu_show( 0 );
		}
	} elsif( menu.settings.show ){
		if( menu.settings.active_sel_state == 0 ){
			HORIS_adahrs.units_menu_show( 1 );
			HORIS_adahrs.settings_menu_show( 0 );
		} else {
			screen.log.write("HORIS: Requested Settings Page Not Yet Implemented");
		}
	} elsif( menu.units.show ){
		if( ( menu.units.active_sel_state + 1 ) != menu.units.active_change_state ){
			menu.units.active_change.setIntValue( menu.units.active_sel_state + 1 );
		} else {
			menu.units.active_change.setIntValue( 0 );
		}
		HORIS_adahrs.units_on_active_change();
	}
}

var main_knob_long = func {
	# enter main menu
	if( !menu.main.show and !menu.settings.show and !menu.units.show ){
		HORIS_adahrs.main_menu_show( 1 );
	}
}

setlistener( menu.units.properties[0], func {
	HORIS_adahrs.update_unit( 0 );
});
setlistener( menu.units.properties[1], func {
	HORIS_adahrs.update_unit( 1 );
});
setlistener( menu.units.properties[2], func {
	HORIS_adahrs.update_unit( 2 );
});
setlistener( menu.units.properties[3], func {
	HORIS_adahrs.update_unit( 3 );
});
setlistener( menu.units.properties[4], func {
	HORIS_adahrs.update_unit( 4 );
});
setlistener( menu.units.properties[5], func {
	HORIS_adahrs.update_unit( 5 );
});
setlistener( menu.units.properties[6], func {
	HORIS_adahrs.update_unit( 6 );
});

var main_knob_button = button.new( 3, main_knob_short, main_knob_long, func return ( volts.getDoubleValue() > 10 ) ) # manual: long = "a few seconds" (p.44)
