# EXPORT : functions ending by export are called from xml
# CRON : functions ending by cron are called from timer
# SCHEDULE : functions ending by schedule are called from cron
# HUMAN : functions ending by human are called by artificial intelligence



# This file contains checklist tasks.


# ===============
# VIRTUAL COPILOT
# ===============

Virtualcopilot = {};

Virtualcopilot.new = func {
   var obj = { parents : [Virtualcopilot,Virtualcrew.new("/systems/copilot")],

               airbleedsystem : nil,
               autopilotsystem : nil,
               electricalsystem : nil,
               flightsystem : nil,
               hydraulicsystem : nil,
               mwssystem : nil,
               voicecrew : nil,
 
               phase : Voicephase.new("/systems/copilot"),
               sensor : Voicesensor.new("/systems/copilot"),
               captain : AsynchronousCrew.new("/systems/copilot"),
               copilot : AsynchronousCrew.new("/systems/copilot"),
               enginetask : CopilotEngineTask.new(),
               flighttask : FlightTask.new(),
               geartask : GearTask.new(),
               groundtask : GroundTask.new(),
               lightingtask : CopilotLightingTask.new(),
               mwstask : CopilotMwsTask.new(),
               navigationtask : CopilotNavigationTask.new(),
               nosetask : NoseTask.new(),
               pilottask : PilotTask.new(),

               FUELSEC : 30.0,
               CRUISESEC : 10.0,
               TAKEOFFSEC : 5.0,

               timercopilot : nil,
               timercopilotslow : nil,
               rates : 0.0,

               pilotincommand : constant.FALSE
         };

   obj.init();

   return obj;
};

Virtualcopilot.init = func {
   me.rates = me.TAKEOFFSEC;
   me.timercopilot = maketimer(me.rates, me, me.schedule);
   me.timercopilot.simulatedTime = 1;
   me.timercopilot.singleShot = 1;

   me.timercopilotslow = maketimer(me.FUELSEC, me, me.slowschedule);
   me.timercopilotslow.simulatedTime = 1;
   me.timercopilotslow.singleShot = 1;
}

Virtualcopilot.set_relation = func( airbleed, autopilot, electrical, flight, hydraulic, lighting, mws, voice ) {
   me.airbleedsystem = airbleed;
   me.autopilotsystem = autopilot;
   me.electricalsystem = electrical;
   me.flightsystem = flight;
   me.hydraulicsystem = hydraulic;
   me.mwssystem = mws;
   me.voicecrew = voice;

   me.copilot.set_relation( autopilot, lighting );
   
   me.enginetask.set_relation( me );
   me.flighttask.set_relation( me, me.captain, flight, voice );
   me.geartask.set_relation( me, hydraulic, voice );
   me.groundtask.set_relation( me, airbleed, electrical, voice );
   me.lightingtask.set_relation( me, me.captain, voice );
   me.mwstask.set_relation( me, me.captain, mws, voice );
   me.navigationtask.set_relation( me );
   me.nosetask.set_relation( me );
   me.pilottask.set_relation( me, autopilot );
}

Virtualcopilot.stateexport = func( targetstate ) {
    me.phase.set_rates( me.CRUISESEC );
    me.phase.schedule();
    me.sensor.schedule();


    me.set_state( constant.TRUE );
    
    if( targetstate == "takeoff" ) {
        me.beforetakeoff();
    }
    
    elsif( targetstate == "taxi" ) {
        me.afterstart();
        me.taxi();
    }
    
    elsif( targetstate == "climb" ) {
        me.aftertakeoff();
        me.climb();
    }
    
    elsif( targetstate == "cruise" ) {
        me.aftertakeoff();
        me.climb();
        me.transsonic();
    }
    
    elsif( targetstate == "descent" ) {
        me.aftertakeoff();
        me.climb();
        me.transsonic();
        me.descent();
    }
    
    elsif( targetstate == "approach" ) {
        me.approach();
    }
    
    elsif( targetstate == "final" ) {
        me.beforelanding();
    }
    
    elsif( targetstate == "parking" ) {
        me.afterlanding();
        me.parking();
    }
    
    elsif( targetstate == "stopover" ) {
        me.afterlanding();
        me.parking();
        me.stopover();
    }

    me.set_state( constant.FALSE );
}

Virtualcopilot.serviceexport = func {
   var serviceable = me.dependency["crew"].getChild("serviceable").getValue();
 
   me.itself["root"].getChild("serviceable").setValue(serviceable);
   
   if( !serviceable ) {
       me.itself["root-ctrl"].getChild("activ").setValue(constant.FALSE);
   }
}

Virtualcopilot.toggleexport = func {
   var launch = constant.FALSE;

   if( me.itself["root"].getChild("serviceable").getValue() ) {
       if( !me.itself["root-ctrl"].getChild("activ").getValue() ) {
           launch = constant.TRUE;
       }
   }

   me.itself["root-ctrl"].getChild("activ").setValue(launch);
       
   if( launch and !me.is_running() ) {
       # must switch again lights
       me.copilot.set_task();

       me.schedule();
       me.slowschedule();
   }
}

Virtualcopilot.veryslowschedule = func {
}

Virtualcopilot.slowschedule = func {
   me.reset();

   me.rates = me.FUELSEC;

   if( me.itself["root"].getChild("serviceable").getValue() ) {
       me.timestamp();
   }

   # run slow
   if( me.itself["root-ctrl"].getChild("activ").getValue() ) {
       me.timercopilotslow.restart(me.rates);
   }
   else {
       me.timercopilotslow.stop();
   }
}

Virtualcopilot.schedule = func {
   me.reset();

   if( me.itself["root"].getChild("serviceable").getValue() ) {
       me.routine();
   }
   else {
       me.rates = me.CRUISESEC;
       me.itself["root"].getChild("activ").setValue(constant.FALSE);
   }

   # run
   if( me.itself["root-ctrl"].getChild("activ").getValue() ) {
       me.set_running();

       me.timercopilot.restart(me.rates);
   }
   else {
       me.timercopilot.stop();
   }
}

Virtualcopilot.fastschedule = func {
   me.reset();

   if( me.itself["root"].getChild("serviceable").getValue() ) {
       me.unexpected();
   }
}

Virtualcopilot.unexpected = func {
   me.pilotincommand = constant.FALSE;

   if( me.itself["root-ctrl"].getChild("activ").getValue() ) {
       me.receive_checklist();

       me.phase.set_rates( me.TAKEOFFSEC );
       me.phase.schedule( constant.TRUE );
       me.sensor.schedule();

       # 4 engines flame out
       me.engine4flameout();

       me.timestamp();
   }

   if( !me.pilotincommand ) {
       me.autopilotsystem.realhuman();
   }

   me.dependency["crew"].getChild("unexpected").setValue(me.pilotincommand);
}

Virtualcopilot.routine = func {
   me.rates = me.CRUISESEC;

   if( me.itself["root-ctrl"].getChild("activ").getValue() ) {
       me.receive_checklist();
       me.receive_emergency();

       # normal procedures
       if ( me.normal() ) {
            me.rates = me.randoms( me.rates );
       }

       me.timestamp();
   }

   me.itself["root"].getChild("activ").setValue(me.is_activ());
}

Virtualcopilot.engine4flameout = func {
   # hold heading and speed, during engine start
   if( !me.phase.is_altitude_approach() ) {
       if( me.autopilotsystem.no_voltage() ) {
           me.pilotincommand = constant.TRUE;

           me.log("no-autopilot");

           me.autopilotsystem.virtualhuman();

           me.pilottask.keepheading();

           me.pilottask.keepspeed();
       }
   }
}

# instrument failures ignored
Virtualcopilot.normal = func {
    me.rates = me.TAKEOFFSEC;

    me.phase.set_rates( me.rates );
    me.phase.schedule();
    me.sensor.schedule();


    # normal
    if( me.checklist.is_beforetakeoff() ) {
        me.set_activ();
        me.beforetakeoff();
        me.rates = me.TAKEOFFSEC;
    }

    elsif( me.checklist.is_taxi() ) {
        me.set_activ();
        me.taxi();
    }

    elsif( me.checklist.is_afterstart() ) {
        me.set_activ();
        me.afterstart();
    }

    elsif( me.checklist.is_pushback() ) {
        me.set_activ();
        me.pushback();
    }

    elsif( me.checklist.is_enginestart() ) {
        me.set_activ();
        me.completed();
    }

    elsif( me.checklist.is_beforestart() ) {
        me.set_activ();
        me.beforestart();
    }

    elsif( me.checklist.is_cockpit() ) {
        me.set_activ();
        me.cockpit();
    }

    elsif( me.checklist.is_preliminary() ) {
        me.completed();
    }

    elsif( me.checklist.is_external() ) {
        me.completed();
    }

    elsif( me.checklist.is_stopover() ) {
        me.set_activ();
        me.stopover();
    }

    elsif( me.checklist.is_parking() ) {
        me.set_activ();
        me.parking();
    }

    elsif( me.checklist.is_afterlanding() ) {
        me.set_activ();
        me.afterlanding();
    }

    elsif( me.checklist.is_beforelanding() ) {
        me.set_activ();
        me.beforelanding();
        me.rates = me.TAKEOFFSEC;
    }

    elsif( me.checklist.is_approach() ) {
        me.set_activ();
        me.approach();
        me.rates = me.TAKEOFFSEC;
    }

    elsif( me.checklist.is_descent() ) {
        me.set_activ();
        me.descent();
    }

    elsif( me.checklist.is_transsonic() ) {
        me.set_activ();
        me.transsonic();
    }

    elsif( me.checklist.is_climb() ) {
        me.set_activ();
        me.climb();
    }

    elsif( me.checklist.is_aftertakeoff() ) {
        me.set_activ();
        me.aftertakeoff();
        me.rates = me.TAKEOFFSEC;
    }


    # emergency
    elsif( me.checklist.is_fourengineflameout() ) {
        me.set_activ();
        me.fourengineflameout();
    }

    elsif( me.checklist.is_fourengineflameoutmach1() ) {
        me.set_activ();
        me.fourengineflameoutmach1();
    }


    me.allways();

    return me.is_activ();
}

Virtualcopilot.get_phase = func {
    return me.phase;
}

Virtualcopilot.get_sensor = func {
    return me.sensor;
}


# ------
# FLIGHT
# ------
Virtualcopilot.allways = func {
    if( !me.phase.is_altitude_approach() ) {
        me.nosetask.nosevisorup();
    }

    if( me.phase.is_altitude_transition() ) {
        me.navigationtask.altimeter();
    }

    me.copilot.do_task( "copilot", me );
}

Virtualcopilot.aftertakeoff = func {
    me.geartask.landinggear( constant.FALSE );
    
    if( me.phase.is_agl_climb() ) {
        me.lightingtask.mainlanding( constant.FALSE );
        me.lightingtask.landingtaxi( constant.FALSE );

        me.mwstask.mwsrecallcaptain();

        me.nosetask.nosevisordown();

        # otherwise disturbing
        me.enginetask.takeoffmonitor( constant.FALSE );

        me.completed();
    }
}

Virtualcopilot.climb = func {
    me.lightingtask.taxiturn( constant.FALSE );

    me.completed();
}

Virtualcopilot.transsonic = func {
    me.completed();
}

Virtualcopilot.descent = func {
    me.completed();
}

Virtualcopilot.approach = func {
    me.lightingtask.taxiturn( constant.TRUE );

    me.nosetask.nosevisordown();

    # relocation in flight
    me.enginetask.takeoffmonitor( constant.FALSE );

    me.completed();
}

Virtualcopilot.beforelanding = func {
    me.geartask.landinggear( constant.TRUE );
    me.nosetask.nosevisor12deg();
    me.geartask.brakelever( constant.FALSE );

    me.lightingtask.mainlanding( constant.TRUE );
    me.lightingtask.landingtaxi( me.dependency["landing-lights"].getChild("taxi").getValue() );

    # relocation in flight
    me.enginetask.takeoffmonitor( constant.FALSE );

    me.completed();
}


# ------
# GROUND
# ------
Virtualcopilot.afterlanding = func {
    me.mwstask.mwsinhibitcaptain();

    me.nosetask.nosevisor5deg();

    me.lightingtask.landingtaxi( constant.TRUE );
    me.lightingtask.mainlanding( constant.FALSE );

    me.completed();
}

Virtualcopilot.parking = func {
    me.lightingtask.mainlanding( constant.FALSE );
    me.lightingtask.landingtaxi( constant.FALSE );
    me.lightingtask.taxiturn( constant.FALSE );

    me.nosetask.nosevisorup();

    for( var i = 0; i < constantaero.NBAUTOPILOTS; i = i+1 ) {
         me.checklist.ins( me, i, constantaero.INSALIGN );
    }

    me.completed();
}

Virtualcopilot.stopover = func {
    me.flighttask.allinverter( constant.FALSE );

    me.completed();
}

Virtualcopilot.cockpit = func {
    me.flighttask.allinverter( constant.TRUE );

    me.completed();
}

Virtualcopilot.beforestart = func {
    me.geartask.brakelever( constant.TRUE );

    if( me.can() ) {
        if( !me.checklist.is_completed() ) {
            if( !me.wait_ground() ) {
                var callsign = me.noinstrument["user"].getValue();

                me.voicecrew.pilotcheck( "clearance", callsign );

                # must wait for ATC answer
                me.done_ground();
            }

            else {
                me.reset_ground();

                me.voicecrew.pilotcheck( "clear" );

                me.completed();
            }
        }
    }
}

Virtualcopilot.pushback = func {
    if( me.sensor.is_inboardengines() ) {
        me.groundtask.grounddisconnect();
        me.geartask.tractor();

        me.completed();
    }
}

Virtualcopilot.afterstart = func {
    me.geartask.steering();
    me.groundtask.grounddisconnect();

    me.flighttask.flightchannelcaptain();

    if( me.has_completed() ) {
        me.voicecrew.pilotcheck( "completed" );
        me.voicecrew.startedexport();
    }
}

Virtualcopilot.taxi = func {
    me.nosetask.nosevisor5deg();

    me.lightingtask.taxiturncaptain( constant.TRUE );
    me.lightingtask.landingtaxicaptain( constant.TRUE );
    me.lightingtask.mainlanding( constant.FALSE );

    me.completed();
}

Virtualcopilot.beforetakeoff = func {
    if( me.checklist.is_startup() ) {
        me.nosetask.nosevisor5deg();
        me.checklist.unset_startup();
    }

    me.lightingtask.taxiturn( constant.TRUE );
    me.lightingtask.mainlanding( constant.TRUE );
    me.lightingtask.landingtaxi( me.dependency["landing-lights"].getChild("taxi").getValue() );

    me.mwstask.mwsrecallcaptain();
    me.mwstask.mwsinhibitcaptain();

    me.enginetask.takeoffmonitor( constant.TRUE );

    me.completed();
}


# ---------
# EMERGENCY
# ---------
Virtualcopilot.fourengineflameout = func {
    me.completed();
}

Virtualcopilot.fourengineflameoutmach1 = func {
    me.completed();
}


# ========
# MWS TASK
# ========

CopilotMwsTask = {};

CopilotMwsTask.new = func {
   var obj = { parents : [CopilotMwsTask,VirtualTask.new(),System.new("/systems/copilot")],
   
               captain : nil,
               mwssystem : nil,
               voicecrew : nil
         };

   obj.init();

   return obj;
};

CopilotMwsTask.init = func {
}

CopilotMwsTask.set_relation = func( crew, captaincrew, mws, voice ) {
    me.crewhuman = crew;
    me.captain = captaincrew;
    me.mwssystem = mws;
    me.voicecrew = voice;
}

CopilotMwsTask.mwsinhibitcaptain = func {
    if( me.can() ) {
        if( !me.dependency["mws"].getChild("inhibit").getValue() ) {
            if( me.captain.is_busy() or me.is_state() ) {
                me.dependency["mws"].getChild("inhibit").setValue(constant.TRUE);
                me.toggleclick("inhibit");
                me.voicecrew.captaincheck( "inhibit" );
            }

            else {
                me.done_crew("not-inhibit");
                me.voicecrew.engineercheck( "inhibit" );
            }
        }
    }
}

CopilotMwsTask.mwsrecallcaptain = func {
    if( me.can() ) {
        if( !me.get_checklist().is_recall() ) {
            if( me.captain.is_busy() or me.is_state() ) {
                me.mwssystem.recallexport();
                me.toggleclick("recall");
                me.voicecrew.captaincheck( "recall" );
            }

            else {
                me.done_crew("not-recall");
                me.voicecrew.engineercheck( "recall" );
            }
        }
    }
}


# ===========
# FLIGHT TASK
# ===========

FlightTask = {};

FlightTask.new = func {
   var obj = { parents : [FlightTask,VirtualTask.new(),System.new("/systems/copilot")],
   
               captain : nil,
               flightsystem : nil,
               voicecrew : nil
         };

   obj.init();

   return obj;
};

FlightTask.init = func {
}

FlightTask.set_relation = func( crew, captaincrew, flight, voice ) {
    me.crewhuman = crew;
    me.captain = captaincrew;
    me.flightsystem = flight;
    me.voicecrew = voice;
}

FlightTask.allinverter = func( value ) {
    me.inverter( "blue", value );
    me.inverter( "green", value );
}

FlightTask.inverter = func( color, value ) {
    if( me.can() ) {
        var path = "inverter-" ~ color;

        if( me.dependency["electric-dc"].getChild(path).getValue() != value ) {
            me.dependency["electric-dc"].getChild(path).setValue(value);
            me.toggleclick(path);
        }
    }
}

FlightTask.is_flightchannel = func {
    var result = constant.TRUE;

    if( me.can() ) {
        if( !me.dependency["channel"].getChild("rudder-blue").getValue() or
            me.dependency["channel"].getChild("rudder-mechanical").getValue() or
            !me.dependency["channel"].getChild("inner-blue").getValue() or
            me.dependency["channel"].getChild("inner-mechanical").getValue() or
            !me.dependency["channel"].getChild("outer-blue").getValue() or
            me.dependency["channel"].getChild("outer-mechanical").getValue() ) {
            result = constant.FALSE;

            me.done_crew("channel-not-blue");
        }

        if( result ) {
            me.reset_crew();
        }
    }

    # captain must reset channels
    return result;
}

FlightTask.flightchannel = func {
    if( me.can() ) {
        if( !me.dependency["channel"].getChild("rudder-blue").getValue() or
            me.dependency["channel"].getChild("rudder-mechanical").getValue() ) {
            me.dependency["channel"].getChild("rudder-blue").setValue(constant.TRUE);
            me.dependency["channel"].getChild("rudder-mechanical").setValue(constant.FALSE);

            me.flightsystem.resetexport();
            me.toggleclick("rudder-channel");
        }

        elsif( !me.dependency["channel"].getChild("inner-blue").getValue() or
            me.dependency["channel"].getChild("inner-mechanical").getValue() ) {
            me.dependency["channel"].getChild("inner-blue").setValue(constant.TRUE);
            me.dependency["channel"].getChild("inner-mechanical").setValue(constant.FALSE);

            me.flightsystem.resetexport();
            me.toggleclick("inner-channel");
        }

        elsif( !me.dependency["channel"].getChild("outer-blue").getValue() or
            me.dependency["channel"].getChild("outer-mechanical").getValue() ) {
            me.dependency["channel"].getChild("outer-blue").setValue(constant.TRUE);
            me.dependency["channel"].getChild("outer-mechanical").setValue(constant.FALSE);

            me.flightsystem.resetexport();
            me.toggleclick("outer-channel");
        }
    }
}

FlightTask.flightchannelcaptain = func {
    if( me.captain.is_busy() or me.is_state() ) {
        var available = me.can();

        me.flightchannel();

        if( available and me.is_flightchannel() ) {
            me.voicecrew.captaincheck( "channel" );
        }
    }

    elsif( !me.is_flightchannel() ) {
        me.voicecrew.pilotcheck( "channel" );
    }
}


# ===========
# ENGINE TASK
# ===========

CopilotEngineTask = {};

CopilotEngineTask.new = func {
   var obj = { parents : [CopilotEngineTask,VirtualTask.new(),System.new("/systems/copilot")]
         };

   obj.init();

   return obj;
};

CopilotEngineTask.init = func {
}

CopilotEngineTask.takeoffmonitor = func( set ) {
    if( me.can() ) {
        var path = "armed";

        if( me.dependency["to-monitor"].getChild(path).getValue() != set ) {
            me.dependency["to-monitor"].getChild(path).setValue( set );
            me.toggleclick("to-monitor");
        }
    }
}


# ==========
# PILOT TASK
# ==========

PilotTask = {};

PilotTask.new = func {
   var obj = { parents : [PilotTask,VirtualTask.new(),System.new("/systems/copilot")],
   
               autopilotsystem : nil,

               VLA41KT : 300.0,
               VLA15KT : 250.0,
               MARGINKT : 25.0,

               FL41FT : 41000.0,
               FL15FT : 15000.0,

               STEPFTPM : 100.0,
               GLIDEFTPM : -1500.0                           # best glide (guess)
         };

   obj.init();

   return obj;
};

PilotTask.init = func {
}

PilotTask.set_relation = func( crew, autopilot ) {
    me.crewhuman = crew;
    me.autopilotsystem = autopilot;
}

PilotTask.keepspeed = func {
    var descentftpm = 0.0;
    
    if( !me.autopilotsystem.is_vertical_speed() ) {
        me.log("vertical-speed");
        me.autopilotsystem.apverticalexport();
        descentftpm = me.GLIDEFTPM;
    }

    # the copilot follows the best glide
    descentftpm = me.adjustglide( descentftpm );  
    me.autopilotsystem.verticalspeed( descentftpm );
}

PilotTask.adjustglide = func( descentftpm ) {
    var minkt = 0;

    if( me.get_phase().is_altitude_above( me.FL41FT ) ) {
        minkt = me.VLA41KT;
    }
    elsif( me.get_phase().is_altitude_above( me.FL15FT ) and me.get_phase().is_altitude_below( me.FL41FT ) ) {
        minkt = me.VLA15KT;
    }
    else {
        minkt = constantaero.V2FULLKT;
    }

    # stay above VLA (lowest allowed speed)
    minkt = minkt + me.MARGINKT;

    if( me.get_phase().is_speed_below( minkt ) ) {
        descentftpm = descentftpm - me.STEPFTPM;
    }
    
    return descentftpm;
}

PilotTask.keepheading = func {
    if( !me.autopilotsystem.is_lock_magnetic() ) {
        me.log("magnetic");
        me.autopilotsystem.apheadingholdexport();
    }
}


# =========
# GEAR TASK
# =========

GearTask = {};

GearTask.new = func {
   var obj = { parents : [GearTask,VirtualTask.new(),System.new("/systems/copilot")],
   
               hydraulicsystem : nil,
               voicecrew : nil
         };

   obj.init();

   return obj;
};

GearTask.init = func {
}

GearTask.set_relation = func( crew, hydraulic, voice ) {
    me.crewhuman = crew;
    me.hydraulicsystem = hydraulic;
    me.voicecrew = voice;
}

GearTask.steering = func {
    if( me.can() ) {
        if( !me.dependency["steering"].getChild("hydraulic").getValue() ) {
            me.dependency["steering"].getChild("hydraulic").setValue( constant.TRUE );
            me.toggleclick("steering");
        }
    }
}

GearTask.brakelever = func( set ) {
    if( me.can() ) {
        if( me.dependency["gear-ctrl"].getChild("brake-parking-lever").getValue() != set ) {
            me.hydraulicsystem.brakesemergencyexport();
            me.toggleclick("brake-parking");
        }
    }
}

GearTask.landinggear = func( landing ) {
    if( me.can() ) {
        if( !landing ) {
            if( me.dependency["gear-ctrl"].getChild("gear-down").getValue() ) {
                if( !me.get_phase().is_agl_below( constantaero.GEARFT ) and me.get_phase().is_speed_above( constantaero.GEARKT ) ) {
                    controls.gearDown(-1);
                    me.toggleclick("gear-up");
                }

                # waits
                else {
                    me.done();
                }
            }

            elsif( me.dependency["gear-ctrl"].getChild("hydraulic").getValue() ) {
                if( me.get_sensor().is_gear_up() ) {
                    controls.gearDown(-1);
                    me.toggleclick("gear-neutral");
                }

                # waits
                else {
                    me.done();
                }
            }
        }

        elsif( !me.dependency["gear-ctrl"].getChild("gear-down").getValue() ) {
            if( me.get_phase().is_agl_landing() and me.get_phase().is_speed_below( constantaero.GEARKT ) ) {
                controls.gearDown(1);
                me.toggleclick("gear-down");
            }

            # waits
            else {
                me.done();
            }
        }
    }
}

GearTask.tractor = func {
    if( me.can() ) {
        # waiting for start of outboard engines
        if( me.dependency["tractor"].getChild("engine14").getValue() ) {
        }

        # from ground interphone, pushback has ended
        elsif( me.dependency["tractor"].getChild("clear").getValue() ) {
            me.dependency["tractor"].getChild("clear").setValue( constant.FALSE );
            # engineer can start outboard engines
            me.dependency["tractor"].getChild("engine14").setValue( constant.TRUE );
            me.brakelever( constant.TRUE );
            me.voicecrew.pilotcheck( "clear" );
        }

        # from ground interphone, pushback has started
        elsif( !me.dependency["tractor"].getChild("pushback").getValue() ) {
            me.dependency["tractor-ctrl"].getChild("pushback").setValue( constant.TRUE );
            me.brakelever( constant.FALSE );
            me.voicecrew.pilotcheck( "pushback" );
        }

        # waiting for end of pushback
        else {
            me.done();
        }
    }
}


# =========
# NOSE TASK
# =========

NoseTask = {};

NoseTask.new = func {
   var obj = { parents : [NoseTask,VirtualTask.new(),System.new("/systems/copilot")],
   
               VISORUP : 0,
               VISORDOWN : 1,

               NOSE5DEG : 2,
               NOSEDOWN : 3,
         };

   obj.init();

   return obj;
};

NoseTask.init = func {
}

NoseTask.nosevisorup = func {
    me.nosevisor( me.VISORUP );
}

NoseTask.nosevisordown = func {
    me.nosevisor( me.VISORDOWN );
}

NoseTask.nosevisor5deg = func {
    me.nosevisor( me.NOSE5DEG );
}

NoseTask.nosevisor12deg = func {
    me.nosevisor( me.NOSEDOWN );
}

NoseTask.nosevisor = func( targetpos ) {
    if( me.can() ) {
        var currentpos = 0;
        var child = nil;


        # not to us to create the property
        child = me.dependency["flaps"].getChild("current-setting");
        if( child == nil ) {
            currentpos = me.VISORUP;
        }
        else {
            currentpos = child.getValue();
        }

        pos = targetpos - currentpos;
        if( pos != 0 ) {
            # - must be up above 270 kt.
            # - down only below 220 kt.
            if( ( targetpos <= me.VISORDOWN and me.get_phase().is_speed_below( constantaero.NOSEKT ) ) or
                ( targetpos > me.VISORDOWN and me.get_phase().is_speed_below( constantaero.GEARKT ) ) ) {
                controls.flapsDown( pos );
                me.toggleclick("nose");
            }

            # waits
            else {
                me.done();
            }
        }
    }
}


# ===============
# NAVIGATION TASK
# ===============

CopilotNavigationTask = {};

CopilotNavigationTask.new = func {
   var obj = { parents : [CopilotNavigationTask,VirtualTask.new(),System.new("/systems/copilot")]
         };

   obj.init();

   return obj;
};

CopilotNavigationTask.init = func {
}

CopilotNavigationTask.altimeter = func {
    if( me.can() ) {
        if( me.dependency["altimeter"].getChild("setting-inhg").getValue() != constantISA.SEA_inhg ) {
            me.dependency["altimeter"].getChild("setting-inhg").setValue(constantISA.SEA_inhg);
            me.toggleclick("altimeter");
        }
    }
}


# =============
# LIGHTING TASK
# =============

CopilotLightingTask = {};

CopilotLightingTask.new = func {
   var obj = { parents : [CopilotLightingTask,VirtualTask.new(),System.new("/systems/copilot")],
   
               captain : nil,
               voicecrew : nil
         };

   obj.init();

   return obj;
};

CopilotLightingTask.init = func {
}

CopilotLightingTask.set_relation = func( crew, captaincrew, voice ) {
    me.crewhuman = crew;
    me.captain = captaincrew;
    me.voicecrew = voice;
}

CopilotLightingTask.mainlanding = func( set ) {
    var path = "";

    # optional in checklist
    set = me.is_light_required( set );

    for( var i=0; i < constantaero.NBAUTOPILOTS; i=i+1 ) {
         path = "main-landing[" ~ i ~ "]/extend";

         if( me.dependency["lighting"].getNode(path).getValue() != set ) {
             if( me.can() ) {
                 me.dependency["lighting"].getNode(path).setValue( set );
                 me.toggleclick("landing-extend-" ~ i);
             }
         }

         path = "main-landing[" ~ i ~ "]/on";

         if( me.dependency["lighting"].getNode(path).getValue() != set ) {
             if( me.can() ) {
                 me.dependency["lighting"].getNode(path).setValue( set );
                 me.toggleclick("landing-on-" ~ i);
             }
         }
    }
}

CopilotLightingTask.landingtaxi = func( set ) {
    var path = "";

    # optional in checklist
    set = me.is_light_required( set );

    for( var i=0; i < constantaero.NBAUTOPILOTS; i=i+1 ) {
         path = "landing-taxi[" ~ i ~ "]/extend";

         if( me.dependency["lighting"].getNode(path).getValue() != set ) {
             if( me.can() ) {
                 me.dependency["lighting"].getNode(path).setValue( set );
                 me.toggleclick("taxi-extend-" ~ i);
             }
         }

         path = "landing-taxi[" ~ i ~ "]/on";

         if( me.dependency["lighting"].getNode(path).getValue() != set ) {
             if( me.can() ) {
                 me.dependency["lighting"].getNode(path).setValue( set );
                 me.toggleclick("taxi-on-" ~ i);
             }
         }
    }
}

CopilotLightingTask.landingtaxicaptain = func( set ) {
    if( me.can() ) {
        if( me.is_landingtaxi() != set ) {
            if( me.is_state() ) {
                # optional
            }
            
            elsif( me.captain.is_busy() ) {
                me.landingtaxi( set );

                if( me.is_landingtaxi() == set ) {
                    me.voicecrew.captaincheck( "landingtaxi" );
                }
            }

            else {
                me.done_crew("not-landingtaxi");
                me.voicecrew.engineercheck( "landingtaxi" );
            }
        }
    }
}

CopilotLightingTask.is_landingtaxi = func {
    var result = constant.TRUE;
    var path = "";

    for( var i=0; i < constantaero.NBAUTOPILOTS; i=i+1 ) {
         path = "landing-taxi[" ~ i ~ "]/extend";

         if( !me.dependency["lighting"].getNode(path).getValue() ) {
             result = constant.FALSE;
             break;
         }

         path = "landing-taxi[" ~ i ~ "]/on";

         if( !me.dependency["lighting"].getNode(path).getValue() ) {
             result = constant.FALSE;
             break;
         }
    }

    return result;
}

CopilotLightingTask.taxiturn = func( set ) {
    var path = "";

    # optional in checklist
    set = me.is_light_required( set );

    for( var i=0; i < constantaero.NBAUTOPILOTS; i=i+1 ) {
         path = "taxi-turn[" ~ i ~ "]/on";

         if( me.dependency["lighting"].getNode(path).getValue() != set ) {
             if( me.can() ) {
                 me.dependency["lighting"].getNode(path).setValue( set );
                 me.toggleclick("taxi-turn-" ~ i);
             }
         }
    }
}

CopilotLightingTask.taxiturncaptain = func( set ) {
    if( me.can() ) {
        if( me.is_taxiturn() != set ) {
            if( me.is_state() ) {
                # optional
            }
            
            elsif( me.captain.is_busy() ) {
                me.taxiturn( set );

                if( me.is_taxiturn() == set ) {
                    me.voicecrew.captaincheck( "taxiturn" );
                }
            }

            else {
                me.done_crew("not-taxiturn");
                me.voicecrew.engineercheck( "taxiturn" );
            }
        }
    }
}

CopilotLightingTask.is_taxiturn = func {
    var result = constant.TRUE;
    var path = "";

    for( var i=0; i < constantaero.NBAUTOPILOTS; i=i+1 ) {
         path = "taxi-turn[" ~ i ~ "]/on";

         if( !me.dependency["lighting"].getNode(path).getValue() ) {
             result = constant.FALSE;
             break;
         }
    }

    return result;
}

CopilotLightingTask.is_light_required = func( set ) {
    var result = set;

    if( !me.dependency["landing-lights"].getChild("set").getValue() ) {
        result = constant.FALSE;
    }
    
    elsif( me.is_state() ) {
        # optional
        result = constant.FALSE;
    }
    
    elsif( me.dependency["landing-lights"].getChild("as-required").getValue() ) {
        if( set ) {
            if( !constant.is_lighting( me.noinstrument["sun"].getValue(), me.noinstrument["altitude"].getValue() ) and
                me.noinstrument["visibility"].getValue() > constant.HAZEMETER ) {
                result = constant.FALSE;
            }
        }
    }
    
    return result;
}


# ===========
# GROUND TASK
# ===========

GroundTask = {};

GroundTask.new = func {
   var obj = { parents : [GroundTask,VirtualTask.new(),System.new("/systems/copilot")],
   
               airbleedsystem : nil,
               electricalsystem : nil,
               voicecrew : nil
         };

   obj.init();

   return obj;
};

GroundTask.init = func {
}

GroundTask.set_relation = func( crew, airbleed, electrical, voice ) {
    me.crewhuman = crew;
    me.airbleedsystem = airbleed;
    me.electricalsystem = electrical;
    me.voicecrew = voice;
}

GroundTask.grounddisconnect = func {
    if( me.can() ) {
        if( me.electricalsystem.has_ground() ) {
            if( !me.wait_ground() ) {
                me.voicecrew.pilotcheck( "disconnect" );

                # must wait for electrical system run (ground)
                me.done_ground();
            }

            else  {
                me.electricalsystem.groundserviceexport();

                me.reset_ground();
                me.done();
            }
        }

        elsif( me.airbleedsystem.has_groundservice() ) {
            if( !me.wait_ground() ) {
                # must wait for air bleed system run (ground)
                me.done_ground();
            }

            else  {
                me.airbleedsystem.groundserviceexport();

                me.reset_ground();
                me.done();
            }
        }

        elsif( me.airbleedsystem.has_reargroundservice() ) {
            if( !me.wait_ground() ) {
                # must wait for temperature system run (ground)
                me.done_ground();
            }

            else  {
                me.airbleedsystem.reargroundserviceexport();

                me.reset_ground();
                me.done();
            }
        }

        elsif( me.dependency["gear-ctrl"].getChild("wheel-chocks").getValue() ) {
            if( !me.wait_ground() ) {
                # must wait for removing of wheel chocks (ground)
                me.done_ground();
            }

            else  {
                me.dependency["gear-ctrl"].getChild("wheel-chocks").setValue( constant.FALSE );

                me.reset_ground();
                me.done();
            }
        }
    }
}
